<?php

namespace PHPMaker2024\cidp24;

use Doctrine\DBAL\ParameterType;
use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Query\QueryBuilder;
use Psr\Http\Message\ServerRequestInterface as Request;
use Psr\Http\Message\ResponseInterface as Response;
use Psr\Container\ContainerInterface;
use Slim\Routing\RouteCollectorProxy;
use Slim\App;
use PhpOffice\PhpSpreadsheet\Style\NumberFormat;
use Closure;

/**
 * Page class
 */
class CurrentRefEdit extends CurrentRef
{
    use MessagesTrait;

    // Page ID
    public $PageID = "edit";

    // Project ID
    public $ProjectID = PROJECT_ID;

    // Page object name
    public $PageObjName = "CurrentRefEdit";

    // View file path
    public $View = null;

    // Title
    public $Title = null; // Title for <title> tag

    // Rendering View
    public $RenderingView = false;

    // CSS class/style
    public $CurrentPageName = "CurrentRefEdit";

    // Page headings
    public $Heading = "";
    public $Subheading = "";
    public $PageHeader;
    public $PageFooter;

    // Page layout
    public $UseLayout = true;

    // Page terminated
    private $terminated = false;

    // Page heading
    public function pageHeading()
    {
        global $Language;
        if ($this->Heading != "") {
            return $this->Heading;
        }
        if (method_exists($this, "tableCaption")) {
            return $this->tableCaption();
        }
        return "";
    }

    // Page subheading
    public function pageSubheading()
    {
        global $Language;
        if ($this->Subheading != "") {
            return $this->Subheading;
        }
        if ($this->TableName) {
            return $Language->phrase($this->PageID);
        }
        return "";
    }

    // Page name
    public function pageName()
    {
        return CurrentPageName();
    }

    // Page URL
    public function pageUrl($withArgs = true)
    {
        $route = GetRoute();
        $args = RemoveXss($route->getArguments());
        if (!$withArgs) {
            foreach ($args as $key => &$val) {
                $val = "";
            }
            unset($val);
        }
        return rtrim(UrlFor($route->getName(), $args), "/") . "?";
    }

    // Show Page Header
    public function showPageHeader()
    {
        $header = $this->PageHeader;
        $this->pageDataRendering($header);
        if ($header != "") { // Header exists, display
            echo '<div id="ew-page-header">' . $header . '</div>';
        }
    }

    // Show Page Footer
    public function showPageFooter()
    {
        $footer = $this->PageFooter;
        $this->pageDataRendered($footer);
        if ($footer != "") { // Footer exists, display
            echo '<div id="ew-page-footer">' . $footer . '</div>';
        }
    }

    // Set field visibility
    public function setVisibility()
    {
        $this->ID->setVisibility();
        $this->PlanYear->setVisibility();
        $this->PlanMonth->setVisibility();
        $this->DaysAfterMonthEnd->setVisibility();
        $this->PlanClosingDate->setVisibility();
        $this->CurrentMonthClosingDate->setVisibility();
    }

    // Constructor
    public function __construct()
    {
        parent::__construct();
        global $Language, $DashboardReport, $DebugTimer, $UserTable;
        $this->TableVar = 'current_ref';
        $this->TableName = 'current_ref';

        // Table CSS class
        $this->TableClass = "table table-striped table-bordered table-hover table-sm ew-desktop-table ew-edit-table";

        // Initialize
        $GLOBALS["Page"] = &$this;

        // Language object
        $Language = Container("app.language");

        // Table object (current_ref)
        if (!isset($GLOBALS["current_ref"]) || $GLOBALS["current_ref"]::class == PROJECT_NAMESPACE . "current_ref") {
            $GLOBALS["current_ref"] = &$this;
        }

        // Table name (for backward compatibility only)
        if (!defined(PROJECT_NAMESPACE . "TABLE_NAME")) {
            define(PROJECT_NAMESPACE . "TABLE_NAME", 'current_ref');
        }

        // Start timer
        $DebugTimer = Container("debug.timer");

        // Debug message
        LoadDebugMessage();

        // Open connection
        $GLOBALS["Conn"] ??= $this->getConnection();

        // User table object
        $UserTable = Container("usertable");
    }

    // Get content from stream
    public function getContents(): string
    {
        global $Response;
        return $Response?->getBody() ?? ob_get_clean();
    }

    // Is lookup
    public function isLookup()
    {
        return SameText(Route(0), Config("API_LOOKUP_ACTION"));
    }

    // Is AutoFill
    public function isAutoFill()
    {
        return $this->isLookup() && SameText(Post("ajax"), "autofill");
    }

    // Is AutoSuggest
    public function isAutoSuggest()
    {
        return $this->isLookup() && SameText(Post("ajax"), "autosuggest");
    }

    // Is modal lookup
    public function isModalLookup()
    {
        return $this->isLookup() && SameText(Post("ajax"), "modal");
    }

    // Is terminated
    public function isTerminated()
    {
        return $this->terminated;
    }

    /**
     * Terminate page
     *
     * @param string $url URL for direction
     * @return void
     */
    public function terminate($url = "")
    {
        if ($this->terminated) {
            return;
        }
        global $TempImages, $DashboardReport, $Response;

        // Page is terminated
        $this->terminated = true;

        // Page Unload event
        if (method_exists($this, "pageUnload")) {
            $this->pageUnload();
        }
        DispatchEvent(new PageUnloadedEvent($this), PageUnloadedEvent::NAME);
        if (!IsApi() && method_exists($this, "pageRedirecting")) {
            $this->pageRedirecting($url);
        }

        // Close connection
        CloseConnections();

        // Return for API
        if (IsApi()) {
            $res = $url === true;
            if (!$res) { // Show response for API
                $ar = array_merge($this->getMessages(), $url ? ["url" => GetUrl($url)] : []);
                WriteJson($ar);
            }
            $this->clearMessages(); // Clear messages for API request
            return;
        } else { // Check if response is JSON
            if (WithJsonResponse()) { // With JSON response
                $this->clearMessages();
                return;
            }
        }

        // Go to URL if specified
        if ($url != "") {
            if (!Config("DEBUG") && ob_get_length()) {
                ob_end_clean();
            }

            // Handle modal response
            if ($this->IsModal) { // Show as modal
                $pageName = GetPageName($url);
                $result = ["url" => GetUrl($url), "modal" => "1"];  // Assume return to modal for simplicity
                if (
                    SameString($pageName, GetPageName($this->getListUrl())) ||
                    SameString($pageName, GetPageName($this->getViewUrl())) ||
                    SameString($pageName, GetPageName(CurrentMasterTable()?->getViewUrl() ?? ""))
                ) { // List / View / Master View page
                    if (!SameString($pageName, GetPageName($this->getListUrl()))) { // Not List page
                        $result["caption"] = $this->getModalCaption($pageName);
                        $result["view"] = SameString($pageName, "CurrentRefView"); // If View page, no primary button
                    } else { // List page
                        $result["error"] = $this->getFailureMessage(); // List page should not be shown as modal => error
                        $this->clearFailureMessage();
                    }
                } else { // Other pages (add messages and then clear messages)
                    $result = array_merge($this->getMessages(), ["modal" => "1"]);
                    $this->clearMessages();
                }
                WriteJson($result);
            } else {
                SaveDebugMessage();
                Redirect(GetUrl($url));
            }
        }
        return; // Return to controller
    }

    // Get records from result set
    protected function getRecordsFromRecordset($rs, $current = false)
    {
        $rows = [];
        if (is_object($rs)) { // Result set
            while ($row = $rs->fetch()) {
                $this->loadRowValues($row); // Set up DbValue/CurrentValue
                $row = $this->getRecordFromArray($row);
                if ($current) {
                    return $row;
                } else {
                    $rows[] = $row;
                }
            }
        } elseif (is_array($rs)) {
            foreach ($rs as $ar) {
                $row = $this->getRecordFromArray($ar);
                if ($current) {
                    return $row;
                } else {
                    $rows[] = $row;
                }
            }
        }
        return $rows;
    }

    // Get record from array
    protected function getRecordFromArray($ar)
    {
        $row = [];
        if (is_array($ar)) {
            foreach ($ar as $fldname => $val) {
                if (array_key_exists($fldname, $this->Fields) && ($this->Fields[$fldname]->Visible || $this->Fields[$fldname]->IsPrimaryKey)) { // Primary key or Visible
                    $fld = &$this->Fields[$fldname];
                    if ($fld->HtmlTag == "FILE") { // Upload field
                        if (EmptyValue($val)) {
                            $row[$fldname] = null;
                        } else {
                            if ($fld->DataType == DataType::BLOB) {
                                $url = FullUrl(GetApiUrl(Config("API_FILE_ACTION") .
                                    "/" . $fld->TableVar . "/" . $fld->Param . "/" . rawurlencode($this->getRecordKeyValue($ar))));
                                $row[$fldname] = ["type" => ContentType($val), "url" => $url, "name" => $fld->Param . ContentExtension($val)];
                            } elseif (!$fld->UploadMultiple || !ContainsString($val, Config("MULTIPLE_UPLOAD_SEPARATOR"))) { // Single file
                                $url = FullUrl(GetApiUrl(Config("API_FILE_ACTION") .
                                    "/" . $fld->TableVar . "/" . Encrypt($fld->physicalUploadPath() . $val)));
                                $row[$fldname] = ["type" => MimeContentType($val), "url" => $url, "name" => $val];
                            } else { // Multiple files
                                $files = explode(Config("MULTIPLE_UPLOAD_SEPARATOR"), $val);
                                $ar = [];
                                foreach ($files as $file) {
                                    $url = FullUrl(GetApiUrl(Config("API_FILE_ACTION") .
                                        "/" . $fld->TableVar . "/" . Encrypt($fld->physicalUploadPath() . $file)));
                                    if (!EmptyValue($file)) {
                                        $ar[] = ["type" => MimeContentType($file), "url" => $url, "name" => $file];
                                    }
                                }
                                $row[$fldname] = $ar;
                            }
                        }
                    } else {
                        $row[$fldname] = $val;
                    }
                }
            }
        }
        return $row;
    }

    // Get record key value from array
    protected function getRecordKeyValue($ar)
    {
        $key = "";
        if (is_array($ar)) {
            $key .= @$ar['ID'];
        }
        return $key;
    }

    /**
     * Hide fields for add/edit
     *
     * @return void
     */
    protected function hideFieldsForAddEdit()
    {
        if ($this->isAdd() || $this->isCopy() || $this->isGridAdd()) {
            $this->ID->Visible = false;
        }
    }

    // Lookup data
    public function lookup(array $req = [], bool $response = true)
    {
        global $Language, $Security;

        // Get lookup object
        $fieldName = $req["field"] ?? null;
        if (!$fieldName) {
            return [];
        }
        $fld = $this->Fields[$fieldName];
        $lookup = $fld->Lookup;
        $name = $req["name"] ?? "";
        if (ContainsString($name, "query_builder_rule")) {
            $lookup->FilterFields = []; // Skip parent fields if any
        }

        // Get lookup parameters
        $lookupType = $req["ajax"] ?? "unknown";
        $pageSize = -1;
        $offset = -1;
        $searchValue = "";
        if (SameText($lookupType, "modal") || SameText($lookupType, "filter")) {
            $searchValue = $req["q"] ?? $req["sv"] ?? "";
            $pageSize = $req["n"] ?? $req["recperpage"] ?? 10;
        } elseif (SameText($lookupType, "autosuggest")) {
            $searchValue = $req["q"] ?? "";
            $pageSize = $req["n"] ?? -1;
            $pageSize = is_numeric($pageSize) ? (int)$pageSize : -1;
            if ($pageSize <= 0) {
                $pageSize = Config("AUTO_SUGGEST_MAX_ENTRIES");
            }
        }
        $start = $req["start"] ?? -1;
        $start = is_numeric($start) ? (int)$start : -1;
        $page = $req["page"] ?? -1;
        $page = is_numeric($page) ? (int)$page : -1;
        $offset = $start >= 0 ? $start : ($page > 0 && $pageSize > 0 ? ($page - 1) * $pageSize : 0);
        $userSelect = Decrypt($req["s"] ?? "");
        $userFilter = Decrypt($req["f"] ?? "");
        $userOrderBy = Decrypt($req["o"] ?? "");
        $keys = $req["keys"] ?? null;
        $lookup->LookupType = $lookupType; // Lookup type
        $lookup->FilterValues = []; // Clear filter values first
        if ($keys !== null) { // Selected records from modal
            if (is_array($keys)) {
                $keys = implode(Config("MULTIPLE_OPTION_SEPARATOR"), $keys);
            }
            $lookup->FilterFields = []; // Skip parent fields if any
            $lookup->FilterValues[] = $keys; // Lookup values
            $pageSize = -1; // Show all records
        } else { // Lookup values
            $lookup->FilterValues[] = $req["v0"] ?? $req["lookupValue"] ?? "";
        }
        $cnt = is_array($lookup->FilterFields) ? count($lookup->FilterFields) : 0;
        for ($i = 1; $i <= $cnt; $i++) {
            $lookup->FilterValues[] = $req["v" . $i] ?? "";
        }
        $lookup->SearchValue = $searchValue;
        $lookup->PageSize = $pageSize;
        $lookup->Offset = $offset;
        if ($userSelect != "") {
            $lookup->UserSelect = $userSelect;
        }
        if ($userFilter != "") {
            $lookup->UserFilter = $userFilter;
        }
        if ($userOrderBy != "") {
            $lookup->UserOrderBy = $userOrderBy;
        }
        return $lookup->toJson($this, $response); // Use settings from current page
    }

    // Properties
    public $FormClassName = "ew-form ew-edit-form overlay-wrapper";
    public $IsModal = false;
    public $IsMobileOrModal = false;
    public $DbMasterFilter;
    public $DbDetailFilter;
    public $HashValue; // Hash Value
    public $DisplayRecords = 1;
    public $StartRecord;
    public $StopRecord;
    public $TotalRecords = 0;
    public $RecordRange = 10;
    public $RecordCount;

    /**
     * Page run
     *
     * @return void
     */
    public function run()
    {
        global $ExportType, $Language, $Security, $CurrentForm, $SkipHeaderFooter;

        // Is modal
        $this->IsModal = ConvertToBool(Param("modal"));
        $this->UseLayout = $this->UseLayout && !$this->IsModal;

        // Use layout
        $this->UseLayout = $this->UseLayout && ConvertToBool(Param(Config("PAGE_LAYOUT"), true));

        // View
        $this->View = Get(Config("VIEW"));

        // Load user profile
        if (IsLoggedIn()) {
            Profile()->setUserName(CurrentUserName())->loadFromStorage();
        }

        // Create form object
        $CurrentForm = new HttpForm();
        $this->CurrentAction = Param("action"); // Set up current action
        $this->setVisibility();

        // Set lookup cache
        if (!in_array($this->PageID, Config("LOOKUP_CACHE_PAGE_IDS"))) {
            $this->setUseLookupCache(false);
        }

        // Global Page Loading event (in userfn*.php)
        DispatchEvent(new PageLoadingEvent($this), PageLoadingEvent::NAME);

        // Page Load event
        if (method_exists($this, "pageLoad")) {
            $this->pageLoad();
        }

        // Hide fields for add/edit
        if (!$this->UseAjaxActions) {
            $this->hideFieldsForAddEdit();
        }
        // Use inline delete
        if ($this->UseAjaxActions) {
            $this->InlineDelete = true;
        }

        // Check modal
        if ($this->IsModal) {
            $SkipHeaderFooter = true;
        }
        $this->IsMobileOrModal = IsMobile() || $this->IsModal;

        // Load record by position
        $loadByPosition = false;
        $loaded = false;
        $postBack = false;

        // Set up current action and primary key
        if (IsApi()) {
            // Load key values
            $loaded = true;
            if (($keyValue = Get("ID") ?? Key(0) ?? Route(2)) !== null) {
                $this->ID->setQueryStringValue($keyValue);
                $this->ID->setOldValue($this->ID->QueryStringValue);
            } elseif (Post("ID") !== null) {
                $this->ID->setFormValue(Post("ID"));
                $this->ID->setOldValue($this->ID->FormValue);
            } else {
                $loaded = false; // Unable to load key
            }

            // Load record
            if ($loaded) {
                $loaded = $this->loadRow();
            }
            if (!$loaded) {
                $this->setFailureMessage($Language->phrase("NoRecord")); // Set no record message
                $this->terminate();
                return;
            }
            $this->CurrentAction = "update"; // Update record directly
            $this->OldKey = $this->getKey(true); // Get from CurrentValue
            $postBack = true;
        } else {
            if (Post("action", "") !== "") {
                $this->CurrentAction = Post("action"); // Get action code
                if (!$this->isShow()) { // Not reload record, handle as postback
                    $postBack = true;
                }

                // Get key from Form
                $this->setKey(Post($this->OldKeyName), $this->isShow());
            } else {
                $this->CurrentAction = "show"; // Default action is display

                // Load key from QueryString
                $loadByQuery = false;
                if (($keyValue = Get("ID") ?? Route("ID")) !== null) {
                    $this->ID->setQueryStringValue($keyValue);
                    $loadByQuery = true;
                } else {
                    $this->ID->CurrentValue = null;
                }
                if (!$loadByQuery || Get(Config("TABLE_START_REC")) !== null || Get(Config("TABLE_PAGE_NUMBER")) !== null) {
                    $loadByPosition = true;
                }
            }

            // Load result set
            if ($this->isShow()) {
                if (!$this->IsModal) { // Normal edit page
                    $this->StartRecord = 1; // Initialize start position
                    $this->Recordset = $this->loadRecordset(); // Load records
                    if ($this->TotalRecords <= 0) { // No record found
                        if ($this->getSuccessMessage() == "" && $this->getFailureMessage() == "") {
                            $this->setFailureMessage($Language->phrase("NoRecord")); // Set no record message
                        }
                        $this->terminate("CurrentRefList"); // Return to list page
                        return;
                    } elseif ($loadByPosition) { // Load record by position
                        $this->setupStartRecord(); // Set up start record position
                        // Point to current record
                        if ($this->StartRecord <= $this->TotalRecords) {
                            $this->fetch($this->StartRecord);
                            // Redirect to correct record
                            $this->loadRowValues($this->CurrentRow);
                            $url = $this->getCurrentUrl();
                            $this->terminate($url);
                            return;
                        }
                    } else { // Match key values
                        if ($this->ID->CurrentValue != null) {
                            while ($this->fetch()) {
                                if (SameString($this->ID->CurrentValue, $this->CurrentRow['ID'])) {
                                    $this->setStartRecordNumber($this->StartRecord); // Save record position
                                    $loaded = true;
                                    break;
                                } else {
                                    $this->StartRecord++;
                                }
                            }
                        }
                    }

                    // Load current row values
                    if ($loaded) {
                        $this->loadRowValues($this->CurrentRow);
                    }
                } else {
                    // Load current record
                    $loaded = $this->loadRow();
                } // End modal checking
                $this->OldKey = $loaded ? $this->getKey(true) : ""; // Get from CurrentValue
            }
        }

        // Process form if post back
        if ($postBack) {
            $this->loadFormValues(); // Get form values
        }

        // Validate form if post back
        if ($postBack) {
            if (!$this->validateForm()) {
                $this->EventCancelled = true; // Event cancelled
                $this->restoreFormValues();
                if (IsApi()) {
                    $this->terminate();
                    return;
                } else {
                    $this->CurrentAction = ""; // Form error, reset action
                }
            }
        }

        // Perform current action
        switch ($this->CurrentAction) {
            case "show": // Get a record to display
                if (!$this->IsModal) { // Normal edit page
                    if (!$loaded) {
                        if ($this->getSuccessMessage() == "" && $this->getFailureMessage() == "") {
                            $this->setFailureMessage($Language->phrase("NoRecord")); // Set no record message
                        }
                        $this->terminate("CurrentRefList"); // Return to list page
                        return;
                    } else {
                    }
                } else { // Modal edit page
                    if (!$loaded) { // Load record based on key
                        if ($this->getFailureMessage() == "") {
                            $this->setFailureMessage($Language->phrase("NoRecord")); // No record found
                        }
                        $this->terminate("CurrentRefList"); // No matching record, return to list
                        return;
                    }
                } // End modal checking
                break;
            case "update": // Update
                $returnUrl = $this->getReturnUrl();
                if (GetPageName($returnUrl) == "CurrentRefList") {
                    $returnUrl = $this->addMasterUrl($returnUrl); // List page, return to List page with correct master key if necessary
                }
                $this->SendEmail = true; // Send email on update success
                if ($this->editRow()) { // Update record based on key
                    if ($this->getSuccessMessage() == "") {
                        $this->setSuccessMessage($Language->phrase("UpdateSuccess")); // Update success
                    }

                    // Handle UseAjaxActions with return page
                    if ($this->IsModal && $this->UseAjaxActions) {
                        $this->IsModal = false;
                        if (GetPageName($returnUrl) != "CurrentRefList") {
                            Container("app.flash")->addMessage("Return-Url", $returnUrl); // Save return URL
                            $returnUrl = "CurrentRefList"; // Return list page content
                        }
                    }
                    if (IsJsonResponse()) {
                        $this->terminate(true);
                        return;
                    } else {
                        $this->terminate($returnUrl); // Return to caller
                        return;
                    }
                } elseif (IsApi()) { // API request, return
                    $this->terminate();
                    return;
                } elseif ($this->IsModal && $this->UseAjaxActions) { // Return JSON error message
                    WriteJson(["success" => false, "validation" => $this->getValidationErrors(), "error" => $this->getFailureMessage()]);
                    $this->clearFailureMessage();
                    $this->terminate();
                    return;
                } elseif ($this->getFailureMessage() == $Language->phrase("NoRecord")) {
                    $this->terminate($returnUrl); // Return to caller
                    return;
                } else {
                    $this->EventCancelled = true; // Event cancelled
                    $this->restoreFormValues(); // Restore form values if update failed
                }
        }

        // Set up Breadcrumb
        $this->setupBreadcrumb();

        // Render the record
        $this->RowType = RowType::EDIT; // Render as Edit
        $this->resetAttributes();
        $this->renderRow();
        if (!$this->IsModal) { // Normal view page
            $this->Pager = new PrevNextPager($this, $this->StartRecord, $this->DisplayRecords, $this->TotalRecords, "", $this->RecordRange, $this->AutoHidePager, false, false);
            $this->Pager->PageNumberName = Config("TABLE_PAGE_NUMBER");
            $this->Pager->PagePhraseId = "Record"; // Show as record
        }

        // Set LoginStatus / Page_Rendering / Page_Render
        if (!IsApi() && !$this->isTerminated()) {
            // Setup login status
            SetupLoginStatus();

            // Pass login status to client side
            SetClientVar("login", LoginStatus());

            // Global Page Rendering event (in userfn*.php)
            DispatchEvent(new PageRenderingEvent($this), PageRenderingEvent::NAME);

            // Page Render event
            if (method_exists($this, "pageRender")) {
                $this->pageRender();
            }

            // Render search option
            if (method_exists($this, "renderSearchOptions")) {
                $this->renderSearchOptions();
            }
        }
    }

    // Get upload files
    protected function getUploadFiles()
    {
        global $CurrentForm, $Language;
    }

    // Load form values
    protected function loadFormValues()
    {
        // Load from form
        global $CurrentForm;
        $validate = !Config("SERVER_VALIDATE");

        // Check field name 'ID' first before field var 'x_ID'
        $val = $CurrentForm->hasValue("ID") ? $CurrentForm->getValue("ID") : $CurrentForm->getValue("x_ID");
        if (!$this->ID->IsDetailKey) {
            $this->ID->setFormValue($val);
        }

        // Check field name 'PlanYear' first before field var 'x_PlanYear'
        $val = $CurrentForm->hasValue("PlanYear") ? $CurrentForm->getValue("PlanYear") : $CurrentForm->getValue("x_PlanYear");
        if (!$this->PlanYear->IsDetailKey) {
            if (IsApi() && $val === null) {
                $this->PlanYear->Visible = false; // Disable update for API request
            } else {
                $this->PlanYear->setFormValue($val, true, $validate);
            }
        }

        // Check field name 'PlanMonth' first before field var 'x_PlanMonth'
        $val = $CurrentForm->hasValue("PlanMonth") ? $CurrentForm->getValue("PlanMonth") : $CurrentForm->getValue("x_PlanMonth");
        if (!$this->PlanMonth->IsDetailKey) {
            if (IsApi() && $val === null) {
                $this->PlanMonth->Visible = false; // Disable update for API request
            } else {
                $this->PlanMonth->setFormValue($val, true, $validate);
            }
        }

        // Check field name 'DaysAfterMonthEnd' first before field var 'x_DaysAfterMonthEnd'
        $val = $CurrentForm->hasValue("DaysAfterMonthEnd") ? $CurrentForm->getValue("DaysAfterMonthEnd") : $CurrentForm->getValue("x_DaysAfterMonthEnd");
        if (!$this->DaysAfterMonthEnd->IsDetailKey) {
            if (IsApi() && $val === null) {
                $this->DaysAfterMonthEnd->Visible = false; // Disable update for API request
            } else {
                $this->DaysAfterMonthEnd->setFormValue($val, true, $validate);
            }
        }

        // Check field name 'PlanClosingDate' first before field var 'x_PlanClosingDate'
        $val = $CurrentForm->hasValue("PlanClosingDate") ? $CurrentForm->getValue("PlanClosingDate") : $CurrentForm->getValue("x_PlanClosingDate");
        if (!$this->PlanClosingDate->IsDetailKey) {
            if (IsApi() && $val === null) {
                $this->PlanClosingDate->Visible = false; // Disable update for API request
            } else {
                $this->PlanClosingDate->setFormValue($val, true, $validate);
            }
            $this->PlanClosingDate->CurrentValue = UnFormatDateTime($this->PlanClosingDate->CurrentValue, $this->PlanClosingDate->formatPattern());
        }

        // Check field name 'CurrentMonthClosingDate' first before field var 'x_CurrentMonthClosingDate'
        $val = $CurrentForm->hasValue("CurrentMonthClosingDate") ? $CurrentForm->getValue("CurrentMonthClosingDate") : $CurrentForm->getValue("x_CurrentMonthClosingDate");
        if (!$this->CurrentMonthClosingDate->IsDetailKey) {
            if (IsApi() && $val === null) {
                $this->CurrentMonthClosingDate->Visible = false; // Disable update for API request
            } else {
                $this->CurrentMonthClosingDate->setFormValue($val, true, $validate);
            }
            $this->CurrentMonthClosingDate->CurrentValue = UnFormatDateTime($this->CurrentMonthClosingDate->CurrentValue, $this->CurrentMonthClosingDate->formatPattern());
        }
    }

    // Restore form values
    public function restoreFormValues()
    {
        global $CurrentForm;
        $this->ID->CurrentValue = $this->ID->FormValue;
        $this->PlanYear->CurrentValue = $this->PlanYear->FormValue;
        $this->PlanMonth->CurrentValue = $this->PlanMonth->FormValue;
        $this->DaysAfterMonthEnd->CurrentValue = $this->DaysAfterMonthEnd->FormValue;
        $this->PlanClosingDate->CurrentValue = $this->PlanClosingDate->FormValue;
        $this->PlanClosingDate->CurrentValue = UnFormatDateTime($this->PlanClosingDate->CurrentValue, $this->PlanClosingDate->formatPattern());
        $this->CurrentMonthClosingDate->CurrentValue = $this->CurrentMonthClosingDate->FormValue;
        $this->CurrentMonthClosingDate->CurrentValue = UnFormatDateTime($this->CurrentMonthClosingDate->CurrentValue, $this->CurrentMonthClosingDate->formatPattern());
    }

    /**
     * Load result set
     *
     * @param int $offset Offset
     * @param int $rowcnt Maximum number of rows
     * @return Doctrine\DBAL\Result Result
     */
    public function loadRecordset($offset = -1, $rowcnt = -1)
    {
        // Load List page SQL (QueryBuilder)
        $sql = $this->getListSql();

        // Load result set
        if ($offset > -1) {
            $sql->setFirstResult($offset);
        }
        if ($rowcnt > 0) {
            $sql->setMaxResults($rowcnt);
        }
        $result = $sql->executeQuery();
        if (property_exists($this, "TotalRecords") && $rowcnt < 0) {
            $this->TotalRecords = $result->rowCount();
            if ($this->TotalRecords <= 0) { // Handle database drivers that does not return rowCount()
                $this->TotalRecords = $this->getRecordCount($this->getListSql());
            }
        }

        // Call Recordset Selected event
        $this->recordsetSelected($result);
        return $result;
    }

    /**
     * Load records as associative array
     *
     * @param int $offset Offset
     * @param int $rowcnt Maximum number of rows
     * @return void
     */
    public function loadRows($offset = -1, $rowcnt = -1)
    {
        // Load List page SQL (QueryBuilder)
        $sql = $this->getListSql();

        // Load result set
        if ($offset > -1) {
            $sql->setFirstResult($offset);
        }
        if ($rowcnt > 0) {
            $sql->setMaxResults($rowcnt);
        }
        $result = $sql->executeQuery();
        return $result->fetchAllAssociative();
    }

    /**
     * Load row based on key values
     *
     * @return void
     */
    public function loadRow()
    {
        global $Security, $Language;
        $filter = $this->getRecordFilter();

        // Call Row Selecting event
        $this->rowSelecting($filter);

        // Load SQL based on filter
        $this->CurrentFilter = $filter;
        $sql = $this->getCurrentSql();
        $conn = $this->getConnection();
        $res = false;
        $row = $conn->fetchAssociative($sql);
        if ($row) {
            $res = true;
            $this->loadRowValues($row); // Load row values
        }
        return $res;
    }

    /**
     * Load row values from result set or record
     *
     * @param array $row Record
     * @return void
     */
    public function loadRowValues($row = null)
    {
        $row = is_array($row) ? $row : $this->newRow();

        // Call Row Selected event
        $this->rowSelected($row);
        $this->ID->setDbValue($row['ID']);
        $this->PlanYear->setDbValue($row['PlanYear']);
        $this->PlanMonth->setDbValue($row['PlanMonth']);
        $this->DaysAfterMonthEnd->setDbValue($row['DaysAfterMonthEnd']);
        $this->PlanClosingDate->setDbValue($row['PlanClosingDate']);
        $this->CurrentMonthClosingDate->setDbValue($row['CurrentMonthClosingDate']);
    }

    // Return a row with default values
    protected function newRow()
    {
        $row = [];
        $row['ID'] = $this->ID->DefaultValue;
        $row['PlanYear'] = $this->PlanYear->DefaultValue;
        $row['PlanMonth'] = $this->PlanMonth->DefaultValue;
        $row['DaysAfterMonthEnd'] = $this->DaysAfterMonthEnd->DefaultValue;
        $row['PlanClosingDate'] = $this->PlanClosingDate->DefaultValue;
        $row['CurrentMonthClosingDate'] = $this->CurrentMonthClosingDate->DefaultValue;
        return $row;
    }

    // Load old record
    protected function loadOldRecord()
    {
        // Load old record
        if ($this->OldKey != "") {
            $this->setKey($this->OldKey);
            $this->CurrentFilter = $this->getRecordFilter();
            $sql = $this->getCurrentSql();
            $conn = $this->getConnection();
            $rs = ExecuteQuery($sql, $conn);
            if ($row = $rs->fetch()) {
                $this->loadRowValues($row); // Load row values
                return $row;
            }
        }
        $this->loadRowValues(); // Load default row values
        return null;
    }

    // Render row values based on field settings
    public function renderRow()
    {
        global $Security, $Language, $CurrentLanguage;

        // Initialize URLs

        // Call Row_Rendering event
        $this->rowRendering();

        // Common render codes for all row types

        // ID
        $this->ID->RowCssClass = "row";

        // PlanYear
        $this->PlanYear->RowCssClass = "row";

        // PlanMonth
        $this->PlanMonth->RowCssClass = "row";

        // DaysAfterMonthEnd
        $this->DaysAfterMonthEnd->RowCssClass = "row";

        // PlanClosingDate
        $this->PlanClosingDate->RowCssClass = "row";

        // CurrentMonthClosingDate
        $this->CurrentMonthClosingDate->RowCssClass = "row";

        // View row
        if ($this->RowType == RowType::VIEW) {
            // ID
            $this->ID->ViewValue = $this->ID->CurrentValue;

            // PlanYear
            $this->PlanYear->ViewValue = $this->PlanYear->CurrentValue;

            // PlanMonth
            $this->PlanMonth->ViewValue = $this->PlanMonth->CurrentValue;

            // DaysAfterMonthEnd
            $this->DaysAfterMonthEnd->ViewValue = $this->DaysAfterMonthEnd->CurrentValue;

            // PlanClosingDate
            $this->PlanClosingDate->ViewValue = $this->PlanClosingDate->CurrentValue;
            $this->PlanClosingDate->ViewValue = FormatDateTime($this->PlanClosingDate->ViewValue, $this->PlanClosingDate->formatPattern());

            // CurrentMonthClosingDate
            $this->CurrentMonthClosingDate->ViewValue = $this->CurrentMonthClosingDate->CurrentValue;
            $this->CurrentMonthClosingDate->ViewValue = FormatDateTime($this->CurrentMonthClosingDate->ViewValue, $this->CurrentMonthClosingDate->formatPattern());

            // ID
            $this->ID->HrefValue = "";

            // PlanYear
            $this->PlanYear->HrefValue = "";

            // PlanMonth
            $this->PlanMonth->HrefValue = "";

            // DaysAfterMonthEnd
            $this->DaysAfterMonthEnd->HrefValue = "";

            // PlanClosingDate
            $this->PlanClosingDate->HrefValue = "";

            // CurrentMonthClosingDate
            $this->CurrentMonthClosingDate->HrefValue = "";
        } elseif ($this->RowType == RowType::EDIT) {
            // ID
            $this->ID->setupEditAttributes();
            $this->ID->EditValue = $this->ID->CurrentValue;

            // PlanYear
            $this->PlanYear->setupEditAttributes();
            $this->PlanYear->EditValue = $this->PlanYear->CurrentValue;
            $this->PlanYear->PlaceHolder = RemoveHtml($this->PlanYear->caption());
            if (strval($this->PlanYear->EditValue) != "" && is_numeric($this->PlanYear->EditValue)) {
                $this->PlanYear->EditValue = $this->PlanYear->EditValue;
            }

            // PlanMonth
            $this->PlanMonth->setupEditAttributes();
            $this->PlanMonth->EditValue = $this->PlanMonth->CurrentValue;
            $this->PlanMonth->PlaceHolder = RemoveHtml($this->PlanMonth->caption());
            if (strval($this->PlanMonth->EditValue) != "" && is_numeric($this->PlanMonth->EditValue)) {
                $this->PlanMonth->EditValue = $this->PlanMonth->EditValue;
            }

            // DaysAfterMonthEnd
            $this->DaysAfterMonthEnd->setupEditAttributes();
            $this->DaysAfterMonthEnd->EditValue = $this->DaysAfterMonthEnd->CurrentValue;
            $this->DaysAfterMonthEnd->PlaceHolder = RemoveHtml($this->DaysAfterMonthEnd->caption());
            if (strval($this->DaysAfterMonthEnd->EditValue) != "" && is_numeric($this->DaysAfterMonthEnd->EditValue)) {
                $this->DaysAfterMonthEnd->EditValue = $this->DaysAfterMonthEnd->EditValue;
            }

            // PlanClosingDate
            $this->PlanClosingDate->setupEditAttributes();
            $this->PlanClosingDate->EditValue = HtmlEncode(FormatDateTime($this->PlanClosingDate->CurrentValue, $this->PlanClosingDate->formatPattern()));
            $this->PlanClosingDate->PlaceHolder = RemoveHtml($this->PlanClosingDate->caption());

            // CurrentMonthClosingDate
            $this->CurrentMonthClosingDate->setupEditAttributes();
            $this->CurrentMonthClosingDate->EditValue = HtmlEncode(FormatDateTime($this->CurrentMonthClosingDate->CurrentValue, $this->CurrentMonthClosingDate->formatPattern()));
            $this->CurrentMonthClosingDate->PlaceHolder = RemoveHtml($this->CurrentMonthClosingDate->caption());

            // Edit refer script

            // ID
            $this->ID->HrefValue = "";

            // PlanYear
            $this->PlanYear->HrefValue = "";

            // PlanMonth
            $this->PlanMonth->HrefValue = "";

            // DaysAfterMonthEnd
            $this->DaysAfterMonthEnd->HrefValue = "";

            // PlanClosingDate
            $this->PlanClosingDate->HrefValue = "";

            // CurrentMonthClosingDate
            $this->CurrentMonthClosingDate->HrefValue = "";
        }
        if ($this->RowType == RowType::ADD || $this->RowType == RowType::EDIT || $this->RowType == RowType::SEARCH) { // Add/Edit/Search row
            $this->setupFieldTitles();
        }

        // Call Row Rendered event
        if ($this->RowType != RowType::AGGREGATEINIT) {
            $this->rowRendered();
        }
    }

    // Validate form
    protected function validateForm()
    {
        global $Language, $Security;

        // Check if validation required
        if (!Config("SERVER_VALIDATE")) {
            return true;
        }
        $validateForm = true;
            if ($this->ID->Visible && $this->ID->Required) {
                if (!$this->ID->IsDetailKey && EmptyValue($this->ID->FormValue)) {
                    $this->ID->addErrorMessage(str_replace("%s", $this->ID->caption(), $this->ID->RequiredErrorMessage));
                }
            }
            if ($this->PlanYear->Visible && $this->PlanYear->Required) {
                if (!$this->PlanYear->IsDetailKey && EmptyValue($this->PlanYear->FormValue)) {
                    $this->PlanYear->addErrorMessage(str_replace("%s", $this->PlanYear->caption(), $this->PlanYear->RequiredErrorMessage));
                }
            }
            if (!CheckInteger($this->PlanYear->FormValue)) {
                $this->PlanYear->addErrorMessage($this->PlanYear->getErrorMessage(false));
            }
            if ($this->PlanMonth->Visible && $this->PlanMonth->Required) {
                if (!$this->PlanMonth->IsDetailKey && EmptyValue($this->PlanMonth->FormValue)) {
                    $this->PlanMonth->addErrorMessage(str_replace("%s", $this->PlanMonth->caption(), $this->PlanMonth->RequiredErrorMessage));
                }
            }
            if (!CheckInteger($this->PlanMonth->FormValue)) {
                $this->PlanMonth->addErrorMessage($this->PlanMonth->getErrorMessage(false));
            }
            if ($this->DaysAfterMonthEnd->Visible && $this->DaysAfterMonthEnd->Required) {
                if (!$this->DaysAfterMonthEnd->IsDetailKey && EmptyValue($this->DaysAfterMonthEnd->FormValue)) {
                    $this->DaysAfterMonthEnd->addErrorMessage(str_replace("%s", $this->DaysAfterMonthEnd->caption(), $this->DaysAfterMonthEnd->RequiredErrorMessage));
                }
            }
            if (!CheckInteger($this->DaysAfterMonthEnd->FormValue)) {
                $this->DaysAfterMonthEnd->addErrorMessage($this->DaysAfterMonthEnd->getErrorMessage(false));
            }
            if ($this->PlanClosingDate->Visible && $this->PlanClosingDate->Required) {
                if (!$this->PlanClosingDate->IsDetailKey && EmptyValue($this->PlanClosingDate->FormValue)) {
                    $this->PlanClosingDate->addErrorMessage(str_replace("%s", $this->PlanClosingDate->caption(), $this->PlanClosingDate->RequiredErrorMessage));
                }
            }
            if (!CheckDate($this->PlanClosingDate->FormValue, $this->PlanClosingDate->formatPattern())) {
                $this->PlanClosingDate->addErrorMessage($this->PlanClosingDate->getErrorMessage(false));
            }
            if ($this->CurrentMonthClosingDate->Visible && $this->CurrentMonthClosingDate->Required) {
                if (!$this->CurrentMonthClosingDate->IsDetailKey && EmptyValue($this->CurrentMonthClosingDate->FormValue)) {
                    $this->CurrentMonthClosingDate->addErrorMessage(str_replace("%s", $this->CurrentMonthClosingDate->caption(), $this->CurrentMonthClosingDate->RequiredErrorMessage));
                }
            }
            if (!CheckDate($this->CurrentMonthClosingDate->FormValue, $this->CurrentMonthClosingDate->formatPattern())) {
                $this->CurrentMonthClosingDate->addErrorMessage($this->CurrentMonthClosingDate->getErrorMessage(false));
            }

        // Return validate result
        $validateForm = $validateForm && !$this->hasInvalidFields();

        // Call Form_CustomValidate event
        $formCustomError = "";
        $validateForm = $validateForm && $this->formCustomValidate($formCustomError);
        if ($formCustomError != "") {
            $this->setFailureMessage($formCustomError);
        }
        return $validateForm;
    }

    // Update record based on key values
    protected function editRow()
    {
        global $Security, $Language;
        $oldKeyFilter = $this->getRecordFilter();
        $filter = $this->applyUserIDFilters($oldKeyFilter);
        $conn = $this->getConnection();

        // Load old row
        $this->CurrentFilter = $filter;
        $sql = $this->getCurrentSql();
        $rsold = $conn->fetchAssociative($sql);
        if (!$rsold) {
            $this->setFailureMessage($Language->phrase("NoRecord")); // Set no record message
            return false; // Update Failed
        } else {
            // Load old values
            $this->loadDbValues($rsold);
        }

        // Get new row
        $rsnew = $this->getEditRow($rsold);

        // Update current values
        $this->setCurrentValues($rsnew);

        // Call Row Updating event
        $updateRow = $this->rowUpdating($rsold, $rsnew);
        if ($updateRow) {
            if (count($rsnew) > 0) {
                $this->CurrentFilter = $filter; // Set up current filter
                $editRow = $this->update($rsnew, "", $rsold);
                if (!$editRow && !EmptyValue($this->DbErrorMessage)) { // Show database error
                    $this->setFailureMessage($this->DbErrorMessage);
                }
            } else {
                $editRow = true; // No field to update
            }
            if ($editRow) {
            }
        } else {
            if ($this->getSuccessMessage() != "" || $this->getFailureMessage() != "") {
                // Use the message, do nothing
            } elseif ($this->CancelMessage != "") {
                $this->setFailureMessage($this->CancelMessage);
                $this->CancelMessage = "";
            } else {
                $this->setFailureMessage($Language->phrase("UpdateCancelled"));
            }
            $editRow = false;
        }

        // Call Row_Updated event
        if ($editRow) {
            $this->rowUpdated($rsold, $rsnew);
        }

        // Write JSON response
        if (IsJsonResponse() && $editRow) {
            $row = $this->getRecordsFromRecordset([$rsnew], true);
            $table = $this->TableVar;
            WriteJson(["success" => true, "action" => Config("API_EDIT_ACTION"), $table => $row]);
        }
        return $editRow;
    }

    /**
     * Get edit row
     *
     * @return array
     */
    protected function getEditRow($rsold)
    {
        global $Security;
        $rsnew = [];

        // PlanYear
        $this->PlanYear->setDbValueDef($rsnew, $this->PlanYear->CurrentValue, $this->PlanYear->ReadOnly);

        // PlanMonth
        $this->PlanMonth->setDbValueDef($rsnew, $this->PlanMonth->CurrentValue, $this->PlanMonth->ReadOnly);

        // DaysAfterMonthEnd
        $this->DaysAfterMonthEnd->setDbValueDef($rsnew, $this->DaysAfterMonthEnd->CurrentValue, $this->DaysAfterMonthEnd->ReadOnly);

        // PlanClosingDate
        $this->PlanClosingDate->setDbValueDef($rsnew, UnFormatDateTime($this->PlanClosingDate->CurrentValue, $this->PlanClosingDate->formatPattern()), $this->PlanClosingDate->ReadOnly);

        // CurrentMonthClosingDate
        $this->CurrentMonthClosingDate->setDbValueDef($rsnew, UnFormatDateTime($this->CurrentMonthClosingDate->CurrentValue, $this->CurrentMonthClosingDate->formatPattern()), $this->CurrentMonthClosingDate->ReadOnly);
        return $rsnew;
    }

    /**
     * Restore edit form from row
     * @param array $row Row
     */
    protected function restoreEditFormFromRow($row)
    {
        if (isset($row['PlanYear'])) { // PlanYear
            $this->PlanYear->CurrentValue = $row['PlanYear'];
        }
        if (isset($row['PlanMonth'])) { // PlanMonth
            $this->PlanMonth->CurrentValue = $row['PlanMonth'];
        }
        if (isset($row['DaysAfterMonthEnd'])) { // DaysAfterMonthEnd
            $this->DaysAfterMonthEnd->CurrentValue = $row['DaysAfterMonthEnd'];
        }
        if (isset($row['PlanClosingDate'])) { // PlanClosingDate
            $this->PlanClosingDate->CurrentValue = $row['PlanClosingDate'];
        }
        if (isset($row['CurrentMonthClosingDate'])) { // CurrentMonthClosingDate
            $this->CurrentMonthClosingDate->CurrentValue = $row['CurrentMonthClosingDate'];
        }
    }

    // Set up Breadcrumb
    protected function setupBreadcrumb()
    {
        global $Breadcrumb, $Language;
        $Breadcrumb = new Breadcrumb("index");
        $url = CurrentUrl();
        $Breadcrumb->add("list", $this->TableVar, $this->addMasterUrl("CurrentRefList"), "", $this->TableVar, true);
        $pageId = "edit";
        $Breadcrumb->add("edit", $pageId, $url);
    }

    // Setup lookup options
    public function setupLookupOptions($fld)
    {
        if ($fld->Lookup && $fld->Lookup->Options === null) {
            // Get default connection and filter
            $conn = $this->getConnection();
            $lookupFilter = "";

            // No need to check any more
            $fld->Lookup->Options = [];

            // Set up lookup SQL and connection
            switch ($fld->FieldVar) {
                default:
                    $lookupFilter = "";
                    break;
            }

            // Always call to Lookup->getSql so that user can setup Lookup->Options in Lookup_Selecting server event
            $sql = $fld->Lookup->getSql(false, "", $lookupFilter, $this);

            // Set up lookup cache
            if (!$fld->hasLookupOptions() && $fld->UseLookupCache && $sql != "" && count($fld->Lookup->Options) == 0 && count($fld->Lookup->FilterFields) == 0) {
                $totalCnt = $this->getRecordCount($sql, $conn);
                if ($totalCnt > $fld->LookupCacheCount) { // Total count > cache count, do not cache
                    return;
                }
                $rows = $conn->executeQuery($sql)->fetchAll();
                $ar = [];
                foreach ($rows as $row) {
                    $row = $fld->Lookup->renderViewRow($row, Container($fld->Lookup->LinkTable));
                    $key = $row["lf"];
                    if (IsFloatType($fld->Type)) { // Handle float field
                        $key = (float)$key;
                    }
                    $ar[strval($key)] = $row;
                }
                $fld->Lookup->Options = $ar;
            }
        }
    }

    // Set up starting record parameters
    public function setupStartRecord()
    {
        if ($this->DisplayRecords == 0) {
            return;
        }
        $pageNo = Get(Config("TABLE_PAGE_NUMBER"));
        $startRec = Get(Config("TABLE_START_REC"));
        $infiniteScroll = false;
        $recordNo = $pageNo ?? $startRec; // Record number = page number or start record
        if ($recordNo !== null && is_numeric($recordNo)) {
            $this->StartRecord = $recordNo;
        } else {
            $this->StartRecord = $this->getStartRecordNumber();
        }

        // Check if correct start record counter
        if (!is_numeric($this->StartRecord) || intval($this->StartRecord) <= 0) { // Avoid invalid start record counter
            $this->StartRecord = 1; // Reset start record counter
        } elseif ($this->StartRecord > $this->TotalRecords) { // Avoid starting record > total records
            $this->StartRecord = (int)(($this->TotalRecords - 1) / $this->DisplayRecords) * $this->DisplayRecords + 1; // Point to last page first record
        } elseif (($this->StartRecord - 1) % $this->DisplayRecords != 0) {
            $this->StartRecord = (int)(($this->StartRecord - 1) / $this->DisplayRecords) * $this->DisplayRecords + 1; // Point to page boundary
        }
        if (!$infiniteScroll) {
            $this->setStartRecordNumber($this->StartRecord);
        }
    }

    // Get page count
    public function pageCount() {
        return ceil($this->TotalRecords / $this->DisplayRecords);
    }

    // Page Load event
    public function pageLoad()
    {
        //Log("Page Load");
    }

    // Page Unload event
    public function pageUnload()
    {
        //Log("Page Unload");
    }

    // Page Redirecting event
    public function pageRedirecting(&$url)
    {
        // Example:
        //$url = "your URL";
    }

    // Message Showing event
    // $type = ''|'success'|'failure'|'warning'
    public function messageShowing(&$msg, $type)
    {
        if ($type == "success") {
            //$msg = "your success message";
        } elseif ($type == "failure") {
            //$msg = "your failure message";
        } elseif ($type == "warning") {
            //$msg = "your warning message";
        } else {
            //$msg = "your message";
        }
    }

    // Page Render event
    public function pageRender()
    {
        //Log("Page Render");
    }

    // Page Data Rendering event
    public function pageDataRendering(&$header)
    {
        // Example:
        //$header = "your header";
    }

    // Page Data Rendered event
    public function pageDataRendered(&$footer)
    {
        // Example:
        //$footer = "your footer";
    }

    // Page Breaking event
    public function pageBreaking(&$break, &$content)
    {
        // Example:
        //$break = false; // Skip page break, or
        //$content = "<div style=\"break-after:page;\"></div>"; // Modify page break content
    }

    // Form Custom Validate event
    public function formCustomValidate(&$customError)
    {
        // Return error message in $customError
        return true;
    }
}
