<?php

namespace PHPMaker2024\cidp24;

use Doctrine\DBAL\ParameterType;
use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Query\QueryBuilder;
use Psr\Http\Message\ServerRequestInterface as Request;
use Psr\Http\Message\ResponseInterface as Response;
use Psr\Container\ContainerInterface;
use Slim\Routing\RouteCollectorProxy;
use Slim\App;
use Closure;

/**
 * Page class
 */
class CidpDataSearch extends CidpData
{
    use MessagesTrait;

    // Page ID
    public $PageID = "search";

    // Project ID
    public $ProjectID = PROJECT_ID;

    // Page object name
    public $PageObjName = "CidpDataSearch";

    // View file path
    public $View = null;

    // Title
    public $Title = null; // Title for <title> tag

    // Rendering View
    public $RenderingView = false;

    // CSS class/style
    public $CurrentPageName = "CidpDataSearch";

    // Page headings
    public $Heading = "";
    public $Subheading = "";
    public $PageHeader;
    public $PageFooter;

    // Page layout
    public $UseLayout = true;

    // Page terminated
    private $terminated = false;

    // Page heading
    public function pageHeading()
    {
        global $Language;
        if ($this->Heading != "") {
            return $this->Heading;
        }
        if (method_exists($this, "tableCaption")) {
            return $this->tableCaption();
        }
        return "";
    }

    // Page subheading
    public function pageSubheading()
    {
        global $Language;
        if ($this->Subheading != "") {
            return $this->Subheading;
        }
        if ($this->TableName) {
            return $Language->phrase($this->PageID);
        }
        return "";
    }

    // Page name
    public function pageName()
    {
        return CurrentPageName();
    }

    // Page URL
    public function pageUrl($withArgs = true)
    {
        $route = GetRoute();
        $args = RemoveXss($route->getArguments());
        if (!$withArgs) {
            foreach ($args as $key => &$val) {
                $val = "";
            }
            unset($val);
        }
        return rtrim(UrlFor($route->getName(), $args), "/") . "?";
    }

    // Show Page Header
    public function showPageHeader()
    {
        $header = $this->PageHeader;
        $this->pageDataRendering($header);
        if ($header != "") { // Header exists, display
            echo '<div id="ew-page-header">' . $header . '</div>';
        }
    }

    // Show Page Footer
    public function showPageFooter()
    {
        $footer = $this->PageFooter;
        $this->pageDataRendered($footer);
        if ($footer != "") { // Footer exists, display
            echo '<div id="ew-page-footer">' . $footer . '</div>';
        }
    }

    // Set field visibility
    public function setVisibility()
    {
        $this->DistrictCode->setVisibility();
        $this->DistrictName->setVisibility();
        $this->GRNCode->setVisibility();
        $this->GRN->setVisibility();
        $this->SupplierCode->setVisibility();
        $this->Supplier->setVisibility();
        $this->CampCode->setVisibility();
        $this->Camp->setVisibility();
        $this->VillageCode->setVisibility();
        $this->Village->setVisibility();
        $this->NRC->setVisibility();
        $this->FarmerName->setVisibility();
        $this->DOB->setVisibility();
        $this->Age->setVisibility();
        $this->Group->setVisibility();
        $this->Gender->setVisibility();
        $this->Seedlings->setVisibility();
        $this->Seeds->setVisibility();
        $this->Type->setVisibility();
        $this->DateAllocated->setVisibility();
        $this->ID->setVisibility();
    }

    // Constructor
    public function __construct()
    {
        parent::__construct();
        global $Language, $DashboardReport, $DebugTimer, $UserTable;
        $this->TableVar = 'cidp_data';
        $this->TableName = 'cidp_data';

        // Table CSS class
        $this->TableClass = "table table-striped table-bordered table-hover table-sm ew-desktop-table ew-search-table";

        // Initialize
        $GLOBALS["Page"] = &$this;

        // Language object
        $Language = Container("app.language");

        // Table object (cidp_data)
        if (!isset($GLOBALS["cidp_data"]) || $GLOBALS["cidp_data"]::class == PROJECT_NAMESPACE . "cidp_data") {
            $GLOBALS["cidp_data"] = &$this;
        }

        // Table name (for backward compatibility only)
        if (!defined(PROJECT_NAMESPACE . "TABLE_NAME")) {
            define(PROJECT_NAMESPACE . "TABLE_NAME", 'cidp_data');
        }

        // Start timer
        $DebugTimer = Container("debug.timer");

        // Debug message
        LoadDebugMessage();

        // Open connection
        $GLOBALS["Conn"] ??= $this->getConnection();

        // User table object
        $UserTable = Container("usertable");
    }

    // Get content from stream
    public function getContents(): string
    {
        global $Response;
        return $Response?->getBody() ?? ob_get_clean();
    }

    // Is lookup
    public function isLookup()
    {
        return SameText(Route(0), Config("API_LOOKUP_ACTION"));
    }

    // Is AutoFill
    public function isAutoFill()
    {
        return $this->isLookup() && SameText(Post("ajax"), "autofill");
    }

    // Is AutoSuggest
    public function isAutoSuggest()
    {
        return $this->isLookup() && SameText(Post("ajax"), "autosuggest");
    }

    // Is modal lookup
    public function isModalLookup()
    {
        return $this->isLookup() && SameText(Post("ajax"), "modal");
    }

    // Is terminated
    public function isTerminated()
    {
        return $this->terminated;
    }

    /**
     * Terminate page
     *
     * @param string $url URL for direction
     * @return void
     */
    public function terminate($url = "")
    {
        if ($this->terminated) {
            return;
        }
        global $TempImages, $DashboardReport, $Response;

        // Page is terminated
        $this->terminated = true;

        // Page Unload event
        if (method_exists($this, "pageUnload")) {
            $this->pageUnload();
        }
        DispatchEvent(new PageUnloadedEvent($this), PageUnloadedEvent::NAME);
        if (!IsApi() && method_exists($this, "pageRedirecting")) {
            $this->pageRedirecting($url);
        }

        // Close connection
        CloseConnections();

        // Return for API
        if (IsApi()) {
            $res = $url === true;
            if (!$res) { // Show response for API
                $ar = array_merge($this->getMessages(), $url ? ["url" => GetUrl($url)] : []);
                WriteJson($ar);
            }
            $this->clearMessages(); // Clear messages for API request
            return;
        } else { // Check if response is JSON
            if (WithJsonResponse()) { // With JSON response
                $this->clearMessages();
                return;
            }
        }

        // Go to URL if specified
        if ($url != "") {
            if (!Config("DEBUG") && ob_get_length()) {
                ob_end_clean();
            }

            // Handle modal response
            if ($this->IsModal) { // Show as modal
                $pageName = GetPageName($url);
                $result = ["url" => GetUrl($url), "modal" => "1"];  // Assume return to modal for simplicity
                if (!SameString($pageName, GetPageName($this->getListUrl()))) { // Not List page
                    $result["caption"] = $this->getModalCaption($pageName);
                    $result["view"] = SameString($pageName, "CidpDataView"); // If View page, no primary button
                } else { // List page
                    $result["error"] = $this->getFailureMessage(); // List page should not be shown as modal => error
                    $this->clearFailureMessage();
                }
                WriteJson($result);
            } else {
                SaveDebugMessage();
                Redirect(GetUrl($url));
            }
        }
        return; // Return to controller
    }

    // Get records from result set
    protected function getRecordsFromRecordset($rs, $current = false)
    {
        $rows = [];
        if (is_object($rs)) { // Result set
            while ($row = $rs->fetch()) {
                $this->loadRowValues($row); // Set up DbValue/CurrentValue
                $row = $this->getRecordFromArray($row);
                if ($current) {
                    return $row;
                } else {
                    $rows[] = $row;
                }
            }
        } elseif (is_array($rs)) {
            foreach ($rs as $ar) {
                $row = $this->getRecordFromArray($ar);
                if ($current) {
                    return $row;
                } else {
                    $rows[] = $row;
                }
            }
        }
        return $rows;
    }

    // Get record from array
    protected function getRecordFromArray($ar)
    {
        $row = [];
        if (is_array($ar)) {
            foreach ($ar as $fldname => $val) {
                if (array_key_exists($fldname, $this->Fields) && ($this->Fields[$fldname]->Visible || $this->Fields[$fldname]->IsPrimaryKey)) { // Primary key or Visible
                    $fld = &$this->Fields[$fldname];
                    if ($fld->HtmlTag == "FILE") { // Upload field
                        if (EmptyValue($val)) {
                            $row[$fldname] = null;
                        } else {
                            if ($fld->DataType == DataType::BLOB) {
                                $url = FullUrl(GetApiUrl(Config("API_FILE_ACTION") .
                                    "/" . $fld->TableVar . "/" . $fld->Param . "/" . rawurlencode($this->getRecordKeyValue($ar))));
                                $row[$fldname] = ["type" => ContentType($val), "url" => $url, "name" => $fld->Param . ContentExtension($val)];
                            } elseif (!$fld->UploadMultiple || !ContainsString($val, Config("MULTIPLE_UPLOAD_SEPARATOR"))) { // Single file
                                $url = FullUrl(GetApiUrl(Config("API_FILE_ACTION") .
                                    "/" . $fld->TableVar . "/" . Encrypt($fld->physicalUploadPath() . $val)));
                                $row[$fldname] = ["type" => MimeContentType($val), "url" => $url, "name" => $val];
                            } else { // Multiple files
                                $files = explode(Config("MULTIPLE_UPLOAD_SEPARATOR"), $val);
                                $ar = [];
                                foreach ($files as $file) {
                                    $url = FullUrl(GetApiUrl(Config("API_FILE_ACTION") .
                                        "/" . $fld->TableVar . "/" . Encrypt($fld->physicalUploadPath() . $file)));
                                    if (!EmptyValue($file)) {
                                        $ar[] = ["type" => MimeContentType($file), "url" => $url, "name" => $file];
                                    }
                                }
                                $row[$fldname] = $ar;
                            }
                        }
                    } else {
                        $row[$fldname] = $val;
                    }
                }
            }
        }
        return $row;
    }

    // Get record key value from array
    protected function getRecordKeyValue($ar)
    {
        $key = "";
        if (is_array($ar)) {
            $key .= @$ar['ID'];
        }
        return $key;
    }

    /**
     * Hide fields for add/edit
     *
     * @return void
     */
    protected function hideFieldsForAddEdit()
    {
        if ($this->isAdd() || $this->isCopy() || $this->isGridAdd()) {
            $this->ID->Visible = false;
        }
    }

    // Lookup data
    public function lookup(array $req = [], bool $response = true)
    {
        global $Language, $Security;

        // Get lookup object
        $fieldName = $req["field"] ?? null;
        if (!$fieldName) {
            return [];
        }
        $fld = $this->Fields[$fieldName];
        $lookup = $fld->Lookup;
        $name = $req["name"] ?? "";
        if (ContainsString($name, "query_builder_rule")) {
            $lookup->FilterFields = []; // Skip parent fields if any
        }

        // Get lookup parameters
        $lookupType = $req["ajax"] ?? "unknown";
        $pageSize = -1;
        $offset = -1;
        $searchValue = "";
        if (SameText($lookupType, "modal") || SameText($lookupType, "filter")) {
            $searchValue = $req["q"] ?? $req["sv"] ?? "";
            $pageSize = $req["n"] ?? $req["recperpage"] ?? 10;
        } elseif (SameText($lookupType, "autosuggest")) {
            $searchValue = $req["q"] ?? "";
            $pageSize = $req["n"] ?? -1;
            $pageSize = is_numeric($pageSize) ? (int)$pageSize : -1;
            if ($pageSize <= 0) {
                $pageSize = Config("AUTO_SUGGEST_MAX_ENTRIES");
            }
        }
        $start = $req["start"] ?? -1;
        $start = is_numeric($start) ? (int)$start : -1;
        $page = $req["page"] ?? -1;
        $page = is_numeric($page) ? (int)$page : -1;
        $offset = $start >= 0 ? $start : ($page > 0 && $pageSize > 0 ? ($page - 1) * $pageSize : 0);
        $userSelect = Decrypt($req["s"] ?? "");
        $userFilter = Decrypt($req["f"] ?? "");
        $userOrderBy = Decrypt($req["o"] ?? "");
        $keys = $req["keys"] ?? null;
        $lookup->LookupType = $lookupType; // Lookup type
        $lookup->FilterValues = []; // Clear filter values first
        if ($keys !== null) { // Selected records from modal
            if (is_array($keys)) {
                $keys = implode(Config("MULTIPLE_OPTION_SEPARATOR"), $keys);
            }
            $lookup->FilterFields = []; // Skip parent fields if any
            $lookup->FilterValues[] = $keys; // Lookup values
            $pageSize = -1; // Show all records
        } else { // Lookup values
            $lookup->FilterValues[] = $req["v0"] ?? $req["lookupValue"] ?? "";
        }
        $cnt = is_array($lookup->FilterFields) ? count($lookup->FilterFields) : 0;
        for ($i = 1; $i <= $cnt; $i++) {
            $lookup->FilterValues[] = $req["v" . $i] ?? "";
        }
        $lookup->SearchValue = $searchValue;
        $lookup->PageSize = $pageSize;
        $lookup->Offset = $offset;
        if ($userSelect != "") {
            $lookup->UserSelect = $userSelect;
        }
        if ($userFilter != "") {
            $lookup->UserFilter = $userFilter;
        }
        if ($userOrderBy != "") {
            $lookup->UserOrderBy = $userOrderBy;
        }
        return $lookup->toJson($this, $response); // Use settings from current page
    }
    public $FormClassName = "ew-form ew-search-form";
    public $IsModal = false;
    public $IsMobileOrModal = false;

    /**
     * Page run
     *
     * @return void
     */
    public function run()
    {
        global $ExportType, $Language, $Security, $CurrentForm, $SkipHeaderFooter;

        // Is modal
        $this->IsModal = ConvertToBool(Param("modal"));
        $this->UseLayout = $this->UseLayout && !$this->IsModal;

        // Use layout
        $this->UseLayout = $this->UseLayout && ConvertToBool(Param(Config("PAGE_LAYOUT"), true));

        // View
        $this->View = Get(Config("VIEW"));

        // Load user profile
        if (IsLoggedIn()) {
            Profile()->setUserName(CurrentUserName())->loadFromStorage();
        }

        // Create form object
        $CurrentForm = new HttpForm();
        $this->CurrentAction = Param("action"); // Set up current action
        $this->setVisibility();

        // Set lookup cache
        if (!in_array($this->PageID, Config("LOOKUP_CACHE_PAGE_IDS"))) {
            $this->setUseLookupCache(false);
        }

        // Global Page Loading event (in userfn*.php)
        DispatchEvent(new PageLoadingEvent($this), PageLoadingEvent::NAME);

        // Page Load event
        if (method_exists($this, "pageLoad")) {
            $this->pageLoad();
        }

        // Hide fields for add/edit
        if (!$this->UseAjaxActions) {
            $this->hideFieldsForAddEdit();
        }
        // Use inline delete
        if ($this->UseAjaxActions) {
            $this->InlineDelete = true;
        }

        // Set up lookup cache
        $this->setupLookupOptions($this->DistrictCode);
        $this->setupLookupOptions($this->GRNCode);
        $this->setupLookupOptions($this->SupplierCode);
        $this->setupLookupOptions($this->CampCode);
        $this->setupLookupOptions($this->VillageCode);
        $this->setupLookupOptions($this->Group);
        $this->setupLookupOptions($this->Gender);
        $this->setupLookupOptions($this->Type);

        // Set up Breadcrumb
        $this->setupBreadcrumb();

        // Check modal
        if ($this->IsModal) {
            $SkipHeaderFooter = true;
        }
        $this->IsMobileOrModal = IsMobile() || $this->IsModal;

        // Get action
        $this->CurrentAction = Post("action");
        if ($this->isSearch()) {
            // Build search string for advanced search, remove blank field
            $this->loadSearchValues(); // Get search values
            $srchStr = $this->validateSearch() ? $this->buildAdvancedSearch() : "";
            if ($srchStr != "") {
                $srchStr = "CidpDataList" . "?" . $srchStr;
                // Do not return Json for UseAjaxActions
                if ($this->IsModal && $this->UseAjaxActions) {
                    $this->IsModal = false;
                }
                $this->terminate($srchStr); // Go to list page
                return;
            }
        }

        // Restore search settings from Session
        if (!$this->hasInvalidFields()) {
            $this->loadAdvancedSearch();
        }

        // Render row for search
        $this->RowType = RowType::SEARCH;
        $this->resetAttributes();
        $this->renderRow();

        // Set LoginStatus / Page_Rendering / Page_Render
        if (!IsApi() && !$this->isTerminated()) {
            // Setup login status
            SetupLoginStatus();

            // Pass login status to client side
            SetClientVar("login", LoginStatus());

            // Global Page Rendering event (in userfn*.php)
            DispatchEvent(new PageRenderingEvent($this), PageRenderingEvent::NAME);

            // Page Render event
            if (method_exists($this, "pageRender")) {
                $this->pageRender();
            }

            // Render search option
            if (method_exists($this, "renderSearchOptions")) {
                $this->renderSearchOptions();
            }
        }
    }

    // Build advanced search
    protected function buildAdvancedSearch()
    {
        $srchUrl = "";
        $this->buildSearchUrl($srchUrl, $this->DistrictCode); // DistrictCode
        $this->buildSearchUrl($srchUrl, $this->DistrictName); // DistrictName
        $this->buildSearchUrl($srchUrl, $this->GRNCode); // GRNCode
        $this->buildSearchUrl($srchUrl, $this->GRN); // GRN
        $this->buildSearchUrl($srchUrl, $this->SupplierCode); // SupplierCode
        $this->buildSearchUrl($srchUrl, $this->Supplier); // Supplier
        $this->buildSearchUrl($srchUrl, $this->CampCode); // CampCode
        $this->buildSearchUrl($srchUrl, $this->Camp); // Camp
        $this->buildSearchUrl($srchUrl, $this->VillageCode); // VillageCode
        $this->buildSearchUrl($srchUrl, $this->Village); // Village
        $this->buildSearchUrl($srchUrl, $this->NRC); // NRC
        $this->buildSearchUrl($srchUrl, $this->FarmerName); // FarmerName
        $this->buildSearchUrl($srchUrl, $this->DOB); // DOB
        $this->buildSearchUrl($srchUrl, $this->Age); // Age
        $this->buildSearchUrl($srchUrl, $this->Group); // Group
        $this->buildSearchUrl($srchUrl, $this->Gender); // Gender
        $this->buildSearchUrl($srchUrl, $this->Seedlings); // Seedlings
        $this->buildSearchUrl($srchUrl, $this->Seeds); // Seeds
        $this->buildSearchUrl($srchUrl, $this->Type); // Type
        $this->buildSearchUrl($srchUrl, $this->DateAllocated); // DateAllocated
        $this->buildSearchUrl($srchUrl, $this->ID); // ID
        if ($srchUrl != "") {
            $srchUrl .= "&";
        }
        $srchUrl .= "cmd=search";
        return $srchUrl;
    }

    // Build search URL
    protected function buildSearchUrl(&$url, $fld, $oprOnly = false)
    {
        global $CurrentForm;
        $wrk = "";
        $fldParm = $fld->Param;
        [
            "value" => $fldVal,
            "operator" => $fldOpr,
            "condition" => $fldCond,
            "value2" => $fldVal2,
            "operator2" => $fldOpr2
        ] = $CurrentForm->getSearchValues($fldParm);
        if (is_array($fldVal)) {
            $fldVal = implode(Config("MULTIPLE_OPTION_SEPARATOR"), $fldVal);
        }
        if (is_array($fldVal2)) {
            $fldVal2 = implode(Config("MULTIPLE_OPTION_SEPARATOR"), $fldVal2);
        }
        $fldDataType = $fld->DataType;
        $value = ConvertSearchValue($fldVal, $fldOpr, $fld); // For testing if numeric only
        $value2 = ConvertSearchValue($fldVal2, $fldOpr2, $fld); // For testing if numeric only
        $fldOpr = ConvertSearchOperator($fldOpr, $fld, $value);
        $fldOpr2 = ConvertSearchOperator($fldOpr2, $fld, $value2);
        if (in_array($fldOpr, ["BETWEEN", "NOT BETWEEN"])) {
            $isValidValue = $fldDataType != DataType::NUMBER || $fld->VirtualSearch || IsNumericSearchValue($value, $fldOpr, $fld) && IsNumericSearchValue($value2, $fldOpr2, $fld);
            if ($fldVal != "" && $fldVal2 != "" && $isValidValue) {
                $wrk = "x_" . $fldParm . "=" . urlencode($fldVal) . "&y_" . $fldParm . "=" . urlencode($fldVal2) . "&z_" . $fldParm . "=" . urlencode($fldOpr);
            }
        } else {
            $isValidValue = $fldDataType != DataType::NUMBER || $fld->VirtualSearch || IsNumericSearchValue($value, $fldOpr, $fld);
            if ($fldVal != "" && $isValidValue && IsValidOperator($fldOpr)) {
                $wrk = "x_" . $fldParm . "=" . urlencode($fldVal) . "&z_" . $fldParm . "=" . urlencode($fldOpr);
            } elseif (in_array($fldOpr, ["IS NULL", "IS NOT NULL", "IS EMPTY", "IS NOT EMPTY"]) || ($fldOpr != "" && $oprOnly && IsValidOperator($fldOpr))) {
                $wrk = "z_" . $fldParm . "=" . urlencode($fldOpr);
            }
            $isValidValue = $fldDataType != DataType::NUMBER || $fld->VirtualSearch || IsNumericSearchValue($value2, $fldOpr2, $fld);
            if ($fldVal2 != "" && $isValidValue && IsValidOperator($fldOpr2)) {
                if ($wrk != "") {
                    $wrk .= "&v_" . $fldParm . "=" . urlencode($fldCond) . "&";
                }
                $wrk .= "y_" . $fldParm . "=" . urlencode($fldVal2) . "&w_" . $fldParm . "=" . urlencode($fldOpr2);
            } elseif (in_array($fldOpr2, ["IS NULL", "IS NOT NULL", "IS EMPTY", "IS NOT EMPTY"]) || ($fldOpr2 != "" && $oprOnly && IsValidOperator($fldOpr2))) {
                if ($wrk != "") {
                    $wrk .= "&v_" . $fldParm . "=" . urlencode($fldCond) . "&";
                }
                $wrk .= "w_" . $fldParm . "=" . urlencode($fldOpr2);
            }
        }
        if ($wrk != "") {
            if ($url != "") {
                $url .= "&";
            }
            $url .= $wrk;
        }
    }

    // Load search values for validation
    protected function loadSearchValues()
    {
        // Load search values
        $hasValue = false;

        // DistrictCode
        if ($this->DistrictCode->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // DistrictName
        if ($this->DistrictName->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // GRNCode
        if ($this->GRNCode->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // GRN
        if ($this->GRN->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // SupplierCode
        if ($this->SupplierCode->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // Supplier
        if ($this->Supplier->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // CampCode
        if ($this->CampCode->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // Camp
        if ($this->Camp->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // VillageCode
        if ($this->VillageCode->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // Village
        if ($this->Village->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // NRC
        if ($this->NRC->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // FarmerName
        if ($this->FarmerName->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // DOB
        if ($this->DOB->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // Age
        if ($this->Age->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // Group
        if ($this->Group->AdvancedSearch->get()) {
            $hasValue = true;
        }
        if (is_array($this->Group->AdvancedSearch->SearchValue)) {
            $this->Group->AdvancedSearch->SearchValue = implode(Config("MULTIPLE_OPTION_SEPARATOR"), $this->Group->AdvancedSearch->SearchValue);
        }
        if (is_array($this->Group->AdvancedSearch->SearchValue2)) {
            $this->Group->AdvancedSearch->SearchValue2 = implode(Config("MULTIPLE_OPTION_SEPARATOR"), $this->Group->AdvancedSearch->SearchValue2);
        }

        // Gender
        if ($this->Gender->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // Seedlings
        if ($this->Seedlings->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // Seeds
        if ($this->Seeds->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // Type
        if ($this->Type->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // DateAllocated
        if ($this->DateAllocated->AdvancedSearch->get()) {
            $hasValue = true;
        }

        // ID
        if ($this->ID->AdvancedSearch->get()) {
            $hasValue = true;
        }
        return $hasValue;
    }

    // Render row values based on field settings
    public function renderRow()
    {
        global $Security, $Language, $CurrentLanguage;

        // Initialize URLs

        // Call Row_Rendering event
        $this->rowRendering();

        // Common render codes for all row types

        // DistrictCode
        $this->DistrictCode->RowCssClass = "row";

        // DistrictName
        $this->DistrictName->RowCssClass = "row";

        // GRNCode
        $this->GRNCode->RowCssClass = "row";

        // GRN
        $this->GRN->RowCssClass = "row";

        // SupplierCode
        $this->SupplierCode->RowCssClass = "row";

        // Supplier
        $this->Supplier->RowCssClass = "row";

        // CampCode
        $this->CampCode->RowCssClass = "row";

        // Camp
        $this->Camp->RowCssClass = "row";

        // VillageCode
        $this->VillageCode->RowCssClass = "row";

        // Village
        $this->Village->RowCssClass = "row";

        // NRC
        $this->NRC->RowCssClass = "row";

        // FarmerName
        $this->FarmerName->RowCssClass = "row";

        // DOB
        $this->DOB->RowCssClass = "row";

        // Age
        $this->Age->RowCssClass = "row";

        // Group
        $this->Group->RowCssClass = "row";

        // Gender
        $this->Gender->RowCssClass = "row";

        // Seedlings
        $this->Seedlings->RowCssClass = "row";

        // Seeds
        $this->Seeds->RowCssClass = "row";

        // Type
        $this->Type->RowCssClass = "row";

        // DateAllocated
        $this->DateAllocated->RowCssClass = "row";

        // ID
        $this->ID->RowCssClass = "row";

        // View row
        if ($this->RowType == RowType::VIEW) {
            // DistrictCode
            $curVal = strval($this->DistrictCode->CurrentValue);
            if ($curVal != "") {
                $this->DistrictCode->ViewValue = $this->DistrictCode->lookupCacheOption($curVal);
                if ($this->DistrictCode->ViewValue === null) { // Lookup from database
                    $filterWrk = SearchFilter($this->DistrictCode->Lookup->getTable()->Fields["DistrictCode"]->searchExpression(), "=", $curVal, $this->DistrictCode->Lookup->getTable()->Fields["DistrictCode"]->searchDataType(), "");
                    $sqlWrk = $this->DistrictCode->Lookup->getSql(false, $filterWrk, '', $this, true, true);
                    $conn = Conn();
                    $config = $conn->getConfiguration();
                    $config->setResultCache($this->Cache);
                    $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                    $ari = count($rswrk);
                    if ($ari > 0) { // Lookup values found
                        $arwrk = $this->DistrictCode->Lookup->renderViewRow($rswrk[0]);
                        $this->DistrictCode->ViewValue = $this->DistrictCode->displayValue($arwrk);
                    } else {
                        $this->DistrictCode->ViewValue = $this->DistrictCode->CurrentValue;
                    }
                }
            } else {
                $this->DistrictCode->ViewValue = null;
            }

            // DistrictName
            $this->DistrictName->ViewValue = $this->DistrictName->CurrentValue;

            // GRNCode
            $this->GRNCode->ViewValue = $this->GRNCode->CurrentValue;
            $curVal = strval($this->GRNCode->CurrentValue);
            if ($curVal != "") {
                $this->GRNCode->ViewValue = $this->GRNCode->lookupCacheOption($curVal);
                if ($this->GRNCode->ViewValue === null) { // Lookup from database
                    $filterWrk = SearchFilter($this->GRNCode->Lookup->getTable()->Fields["GRN"]->searchExpression(), "=", $curVal, $this->GRNCode->Lookup->getTable()->Fields["GRN"]->searchDataType(), "");
                    $sqlWrk = $this->GRNCode->Lookup->getSql(false, $filterWrk, '', $this, true, true);
                    $conn = Conn();
                    $config = $conn->getConfiguration();
                    $config->setResultCache($this->Cache);
                    $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                    $ari = count($rswrk);
                    if ($ari > 0) { // Lookup values found
                        $arwrk = $this->GRNCode->Lookup->renderViewRow($rswrk[0]);
                        $this->GRNCode->ViewValue = $this->GRNCode->displayValue($arwrk);
                    } else {
                        $this->GRNCode->ViewValue = $this->GRNCode->CurrentValue;
                    }
                }
            } else {
                $this->GRNCode->ViewValue = null;
            }

            // GRN
            $this->GRN->ViewValue = $this->GRN->CurrentValue;

            // SupplierCode
            $this->SupplierCode->ViewValue = $this->SupplierCode->CurrentValue;
            $curVal = strval($this->SupplierCode->CurrentValue);
            if ($curVal != "") {
                $this->SupplierCode->ViewValue = $this->SupplierCode->lookupCacheOption($curVal);
                if ($this->SupplierCode->ViewValue === null) { // Lookup from database
                    $filterWrk = SearchFilter($this->SupplierCode->Lookup->getTable()->Fields["NurseryCode"]->searchExpression(), "=", $curVal, $this->SupplierCode->Lookup->getTable()->Fields["NurseryCode"]->searchDataType(), "");
                    $sqlWrk = $this->SupplierCode->Lookup->getSql(false, $filterWrk, '', $this, true, true);
                    $conn = Conn();
                    $config = $conn->getConfiguration();
                    $config->setResultCache($this->Cache);
                    $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                    $ari = count($rswrk);
                    if ($ari > 0) { // Lookup values found
                        $arwrk = $this->SupplierCode->Lookup->renderViewRow($rswrk[0]);
                        $this->SupplierCode->ViewValue = $this->SupplierCode->displayValue($arwrk);
                    } else {
                        $this->SupplierCode->ViewValue = $this->SupplierCode->CurrentValue;
                    }
                }
            } else {
                $this->SupplierCode->ViewValue = null;
            }

            // Supplier
            $this->Supplier->ViewValue = $this->Supplier->CurrentValue;

            // CampCode
            $curVal = strval($this->CampCode->CurrentValue);
            if ($curVal != "") {
                $this->CampCode->ViewValue = $this->CampCode->lookupCacheOption($curVal);
                if ($this->CampCode->ViewValue === null) { // Lookup from database
                    $filterWrk = SearchFilter($this->CampCode->Lookup->getTable()->Fields["CampCode"]->searchExpression(), "=", $curVal, $this->CampCode->Lookup->getTable()->Fields["CampCode"]->searchDataType(), "");
                    $sqlWrk = $this->CampCode->Lookup->getSql(false, $filterWrk, '', $this, true, true);
                    $conn = Conn();
                    $config = $conn->getConfiguration();
                    $config->setResultCache($this->Cache);
                    $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                    $ari = count($rswrk);
                    if ($ari > 0) { // Lookup values found
                        $arwrk = $this->CampCode->Lookup->renderViewRow($rswrk[0]);
                        $this->CampCode->ViewValue = $this->CampCode->displayValue($arwrk);
                    } else {
                        $this->CampCode->ViewValue = $this->CampCode->CurrentValue;
                    }
                }
            } else {
                $this->CampCode->ViewValue = null;
            }

            // Camp
            $this->Camp->ViewValue = $this->Camp->CurrentValue;

            // VillageCode
            $curVal = strval($this->VillageCode->CurrentValue);
            if ($curVal != "") {
                $this->VillageCode->ViewValue = $this->VillageCode->lookupCacheOption($curVal);
                if ($this->VillageCode->ViewValue === null) { // Lookup from database
                    $filterWrk = SearchFilter($this->VillageCode->Lookup->getTable()->Fields["VillageCode"]->searchExpression(), "=", $curVal, $this->VillageCode->Lookup->getTable()->Fields["VillageCode"]->searchDataType(), "");
                    $sqlWrk = $this->VillageCode->Lookup->getSql(false, $filterWrk, '', $this, true, true);
                    $conn = Conn();
                    $config = $conn->getConfiguration();
                    $config->setResultCache($this->Cache);
                    $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                    $ari = count($rswrk);
                    if ($ari > 0) { // Lookup values found
                        $arwrk = $this->VillageCode->Lookup->renderViewRow($rswrk[0]);
                        $this->VillageCode->ViewValue = $this->VillageCode->displayValue($arwrk);
                    } else {
                        $this->VillageCode->ViewValue = $this->VillageCode->CurrentValue;
                    }
                }
            } else {
                $this->VillageCode->ViewValue = null;
            }

            // Village
            $this->Village->ViewValue = $this->Village->CurrentValue;

            // NRC
            $this->NRC->ViewValue = $this->NRC->CurrentValue;

            // FarmerName
            $this->FarmerName->ViewValue = $this->FarmerName->CurrentValue;

            // DOB
            $this->DOB->ViewValue = $this->DOB->CurrentValue;

            // Age
            $this->Age->ViewValue = $this->Age->CurrentValue;
            $this->Age->ViewValue = FormatNumber($this->Age->ViewValue, $this->Age->formatPattern());

            // Group
            if (strval($this->Group->CurrentValue) != "") {
                $this->Group->ViewValue = new OptionValues();
                $arwrk = explode(Config("MULTIPLE_OPTION_SEPARATOR"), strval($this->Group->CurrentValue));
                $cnt = count($arwrk);
                for ($ari = 0; $ari < $cnt; $ari++)
                    $this->Group->ViewValue->add($this->Group->optionCaption(trim($arwrk[$ari])));
            } else {
                $this->Group->ViewValue = null;
            }

            // Gender
            $curVal = strval($this->Gender->CurrentValue);
            if ($curVal != "") {
                $this->Gender->ViewValue = $this->Gender->lookupCacheOption($curVal);
                if ($this->Gender->ViewValue === null) { // Lookup from database
                    $filterWrk = SearchFilter($this->Gender->Lookup->getTable()->Fields["Gender"]->searchExpression(), "=", $curVal, $this->Gender->Lookup->getTable()->Fields["Gender"]->searchDataType(), "");
                    $sqlWrk = $this->Gender->Lookup->getSql(false, $filterWrk, '', $this, true, true);
                    $conn = Conn();
                    $config = $conn->getConfiguration();
                    $config->setResultCache($this->Cache);
                    $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                    $ari = count($rswrk);
                    if ($ari > 0) { // Lookup values found
                        $arwrk = $this->Gender->Lookup->renderViewRow($rswrk[0]);
                        $this->Gender->ViewValue = $this->Gender->displayValue($arwrk);
                    } else {
                        $this->Gender->ViewValue = $this->Gender->CurrentValue;
                    }
                }
            } else {
                $this->Gender->ViewValue = null;
            }

            // Seedlings
            $this->Seedlings->ViewValue = $this->Seedlings->CurrentValue;
            $this->Seedlings->ViewValue = FormatNumber($this->Seedlings->ViewValue, $this->Seedlings->formatPattern());
            $this->Seedlings->CellCssStyle .= "text-align: right;";

            // Seeds
            $this->Seeds->ViewValue = $this->Seeds->CurrentValue;
            $this->Seeds->ViewValue = FormatNumber($this->Seeds->ViewValue, $this->Seeds->formatPattern());
            $this->Seeds->CellCssStyle .= "text-align: right;";

            // Type
            $curVal = strval($this->Type->CurrentValue);
            if ($curVal != "") {
                $this->Type->ViewValue = $this->Type->lookupCacheOption($curVal);
                if ($this->Type->ViewValue === null) { // Lookup from database
                    $filterWrk = SearchFilter($this->Type->Lookup->getTable()->Fields["SeedVariety"]->searchExpression(), "=", $curVal, $this->Type->Lookup->getTable()->Fields["SeedVariety"]->searchDataType(), "");
                    $sqlWrk = $this->Type->Lookup->getSql(false, $filterWrk, '', $this, true, true);
                    $conn = Conn();
                    $config = $conn->getConfiguration();
                    $config->setResultCache($this->Cache);
                    $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                    $ari = count($rswrk);
                    if ($ari > 0) { // Lookup values found
                        $arwrk = $this->Type->Lookup->renderViewRow($rswrk[0]);
                        $this->Type->ViewValue = $this->Type->displayValue($arwrk);
                    } else {
                        $this->Type->ViewValue = $this->Type->CurrentValue;
                    }
                }
            } else {
                $this->Type->ViewValue = null;
            }

            // DateAllocated
            $this->DateAllocated->ViewValue = $this->DateAllocated->CurrentValue;

            // ID
            $this->ID->ViewValue = $this->ID->CurrentValue;

            // DistrictCode
            $this->DistrictCode->HrefValue = "";
            $this->DistrictCode->TooltipValue = "";

            // DistrictName
            $this->DistrictName->HrefValue = "";
            $this->DistrictName->TooltipValue = "";

            // GRNCode
            $this->GRNCode->HrefValue = "";
            $this->GRNCode->TooltipValue = "";

            // GRN
            $this->GRN->HrefValue = "";
            $this->GRN->TooltipValue = "";

            // SupplierCode
            $this->SupplierCode->HrefValue = "";
            $this->SupplierCode->TooltipValue = "";

            // Supplier
            $this->Supplier->HrefValue = "";
            $this->Supplier->TooltipValue = "";

            // CampCode
            $this->CampCode->HrefValue = "";
            $this->CampCode->TooltipValue = "";

            // Camp
            $this->Camp->HrefValue = "";
            $this->Camp->TooltipValue = "";

            // VillageCode
            $this->VillageCode->HrefValue = "";
            $this->VillageCode->TooltipValue = "";

            // Village
            $this->Village->HrefValue = "";
            $this->Village->TooltipValue = "";

            // NRC
            $this->NRC->HrefValue = "";
            $this->NRC->TooltipValue = "";

            // FarmerName
            $this->FarmerName->HrefValue = "";
            $this->FarmerName->TooltipValue = "";

            // DOB
            $this->DOB->HrefValue = "";
            $this->DOB->TooltipValue = "";

            // Age
            $this->Age->HrefValue = "";
            $this->Age->TooltipValue = "";

            // Group
            $this->Group->HrefValue = "";
            $this->Group->TooltipValue = "";

            // Gender
            $this->Gender->HrefValue = "";
            $this->Gender->TooltipValue = "";

            // Seedlings
            $this->Seedlings->HrefValue = "";
            $this->Seedlings->TooltipValue = "";

            // Seeds
            $this->Seeds->HrefValue = "";
            $this->Seeds->TooltipValue = "";

            // Type
            $this->Type->HrefValue = "";
            $this->Type->TooltipValue = "";

            // DateAllocated
            $this->DateAllocated->HrefValue = "";
            $this->DateAllocated->TooltipValue = "";

            // ID
            $this->ID->HrefValue = "";
            $this->ID->TooltipValue = "";
        } elseif ($this->RowType == RowType::SEARCH) {
            // DistrictCode
            $curVal = trim(strval($this->DistrictCode->AdvancedSearch->SearchValue));
            if ($curVal != "") {
                $this->DistrictCode->AdvancedSearch->ViewValue = $this->DistrictCode->lookupCacheOption($curVal);
            } else {
                $this->DistrictCode->AdvancedSearch->ViewValue = $this->DistrictCode->Lookup !== null && is_array($this->DistrictCode->lookupOptions()) && count($this->DistrictCode->lookupOptions()) > 0 ? $curVal : null;
            }
            if ($this->DistrictCode->AdvancedSearch->ViewValue !== null) { // Load from cache
                $this->DistrictCode->EditValue = array_values($this->DistrictCode->lookupOptions());
                if ($this->DistrictCode->AdvancedSearch->ViewValue == "") {
                    $this->DistrictCode->AdvancedSearch->ViewValue = $Language->phrase("PleaseSelect");
                }
            } else { // Lookup from database
                if ($curVal == "") {
                    $filterWrk = "0=1";
                } else {
                    $filterWrk = SearchFilter($this->DistrictCode->Lookup->getTable()->Fields["DistrictCode"]->searchExpression(), "=", $this->DistrictCode->AdvancedSearch->SearchValue, $this->DistrictCode->Lookup->getTable()->Fields["DistrictCode"]->searchDataType(), "");
                }
                $sqlWrk = $this->DistrictCode->Lookup->getSql(true, $filterWrk, '', $this, false, true);
                $conn = Conn();
                $config = $conn->getConfiguration();
                $config->setResultCache($this->Cache);
                $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                $ari = count($rswrk);
                if ($ari > 0) { // Lookup values found
                    $arwrk = $this->DistrictCode->Lookup->renderViewRow($rswrk[0]);
                    $this->DistrictCode->AdvancedSearch->ViewValue = $this->DistrictCode->displayValue($arwrk);
                } else {
                    $this->DistrictCode->AdvancedSearch->ViewValue = $Language->phrase("PleaseSelect");
                }
                $arwrk = $rswrk;
                $this->DistrictCode->EditValue = $arwrk;
            }
            $this->DistrictCode->PlaceHolder = RemoveHtml($this->DistrictCode->caption());

            // DistrictName
            $this->DistrictName->setupEditAttributes();
            if (!$this->DistrictName->Raw) {
                $this->DistrictName->AdvancedSearch->SearchValue = HtmlDecode($this->DistrictName->AdvancedSearch->SearchValue);
            }
            $this->DistrictName->EditValue = HtmlEncode($this->DistrictName->AdvancedSearch->SearchValue);
            $this->DistrictName->PlaceHolder = RemoveHtml($this->DistrictName->caption());

            // GRNCode
            $this->GRNCode->setupEditAttributes();
            if (!$this->GRNCode->Raw) {
                $this->GRNCode->AdvancedSearch->SearchValue = HtmlDecode($this->GRNCode->AdvancedSearch->SearchValue);
            }
            $this->GRNCode->EditValue = HtmlEncode($this->GRNCode->AdvancedSearch->SearchValue);
            $curVal = strval($this->GRNCode->AdvancedSearch->SearchValue);
            if ($curVal != "") {
                $this->GRNCode->EditValue = $this->GRNCode->lookupCacheOption($curVal);
                if ($this->GRNCode->EditValue === null) { // Lookup from database
                    $filterWrk = SearchFilter($this->GRNCode->Lookup->getTable()->Fields["GRN"]->searchExpression(), "=", $curVal, $this->GRNCode->Lookup->getTable()->Fields["GRN"]->searchDataType(), "");
                    $sqlWrk = $this->GRNCode->Lookup->getSql(false, $filterWrk, '', $this, true, true);
                    $conn = Conn();
                    $config = $conn->getConfiguration();
                    $config->setResultCache($this->Cache);
                    $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                    $ari = count($rswrk);
                    if ($ari > 0) { // Lookup values found
                        $arwrk = $this->GRNCode->Lookup->renderViewRow($rswrk[0]);
                        $this->GRNCode->EditValue = $this->GRNCode->displayValue($arwrk);
                    } else {
                        $this->GRNCode->EditValue = HtmlEncode($this->GRNCode->AdvancedSearch->SearchValue);
                    }
                }
            } else {
                $this->GRNCode->EditValue = null;
            }
            $this->GRNCode->PlaceHolder = RemoveHtml($this->GRNCode->caption());

            // GRN
            $this->GRN->setupEditAttributes();
            if (!$this->GRN->Raw) {
                $this->GRN->AdvancedSearch->SearchValue = HtmlDecode($this->GRN->AdvancedSearch->SearchValue);
            }
            $this->GRN->EditValue = HtmlEncode($this->GRN->AdvancedSearch->SearchValue);
            $this->GRN->PlaceHolder = RemoveHtml($this->GRN->caption());

            // SupplierCode
            $this->SupplierCode->setupEditAttributes();
            $this->SupplierCode->EditValue = $this->SupplierCode->AdvancedSearch->SearchValue;
            $curVal = strval($this->SupplierCode->AdvancedSearch->SearchValue);
            if ($curVal != "") {
                $this->SupplierCode->EditValue = $this->SupplierCode->lookupCacheOption($curVal);
                if ($this->SupplierCode->EditValue === null) { // Lookup from database
                    $filterWrk = SearchFilter($this->SupplierCode->Lookup->getTable()->Fields["NurseryCode"]->searchExpression(), "=", $curVal, $this->SupplierCode->Lookup->getTable()->Fields["NurseryCode"]->searchDataType(), "");
                    $sqlWrk = $this->SupplierCode->Lookup->getSql(false, $filterWrk, '', $this, true, true);
                    $conn = Conn();
                    $config = $conn->getConfiguration();
                    $config->setResultCache($this->Cache);
                    $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                    $ari = count($rswrk);
                    if ($ari > 0) { // Lookup values found
                        $arwrk = $this->SupplierCode->Lookup->renderViewRow($rswrk[0]);
                        $this->SupplierCode->EditValue = $this->SupplierCode->displayValue($arwrk);
                    } else {
                        $this->SupplierCode->EditValue = HtmlEncode($this->SupplierCode->AdvancedSearch->SearchValue);
                    }
                }
            } else {
                $this->SupplierCode->EditValue = null;
            }
            $this->SupplierCode->PlaceHolder = RemoveHtml($this->SupplierCode->caption());

            // Supplier
            $this->Supplier->setupEditAttributes();
            if (!$this->Supplier->Raw) {
                $this->Supplier->AdvancedSearch->SearchValue = HtmlDecode($this->Supplier->AdvancedSearch->SearchValue);
            }
            $this->Supplier->EditValue = HtmlEncode($this->Supplier->AdvancedSearch->SearchValue);
            $this->Supplier->PlaceHolder = RemoveHtml($this->Supplier->caption());

            // CampCode
            $curVal = trim(strval($this->CampCode->AdvancedSearch->SearchValue));
            if ($curVal != "") {
                $this->CampCode->AdvancedSearch->ViewValue = $this->CampCode->lookupCacheOption($curVal);
            } else {
                $this->CampCode->AdvancedSearch->ViewValue = $this->CampCode->Lookup !== null && is_array($this->CampCode->lookupOptions()) && count($this->CampCode->lookupOptions()) > 0 ? $curVal : null;
            }
            if ($this->CampCode->AdvancedSearch->ViewValue !== null) { // Load from cache
                $this->CampCode->EditValue = array_values($this->CampCode->lookupOptions());
                if ($this->CampCode->AdvancedSearch->ViewValue == "") {
                    $this->CampCode->AdvancedSearch->ViewValue = $Language->phrase("PleaseSelect");
                }
            } else { // Lookup from database
                if ($curVal == "") {
                    $filterWrk = "0=1";
                } else {
                    $filterWrk = SearchFilter($this->CampCode->Lookup->getTable()->Fields["CampCode"]->searchExpression(), "=", $this->CampCode->AdvancedSearch->SearchValue, $this->CampCode->Lookup->getTable()->Fields["CampCode"]->searchDataType(), "");
                }
                $sqlWrk = $this->CampCode->Lookup->getSql(true, $filterWrk, '', $this, false, true);
                $conn = Conn();
                $config = $conn->getConfiguration();
                $config->setResultCache($this->Cache);
                $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                $ari = count($rswrk);
                if ($ari > 0) { // Lookup values found
                    $arwrk = $this->CampCode->Lookup->renderViewRow($rswrk[0]);
                    $this->CampCode->AdvancedSearch->ViewValue = $this->CampCode->displayValue($arwrk);
                } else {
                    $this->CampCode->AdvancedSearch->ViewValue = $Language->phrase("PleaseSelect");
                }
                $arwrk = $rswrk;
                $this->CampCode->EditValue = $arwrk;
            }
            $this->CampCode->PlaceHolder = RemoveHtml($this->CampCode->caption());

            // Camp
            $this->Camp->setupEditAttributes();
            if (!$this->Camp->Raw) {
                $this->Camp->AdvancedSearch->SearchValue = HtmlDecode($this->Camp->AdvancedSearch->SearchValue);
            }
            $this->Camp->EditValue = HtmlEncode($this->Camp->AdvancedSearch->SearchValue);
            $this->Camp->PlaceHolder = RemoveHtml($this->Camp->caption());

            // VillageCode
            $curVal = trim(strval($this->VillageCode->AdvancedSearch->SearchValue));
            if ($curVal != "") {
                $this->VillageCode->AdvancedSearch->ViewValue = $this->VillageCode->lookupCacheOption($curVal);
            } else {
                $this->VillageCode->AdvancedSearch->ViewValue = $this->VillageCode->Lookup !== null && is_array($this->VillageCode->lookupOptions()) && count($this->VillageCode->lookupOptions()) > 0 ? $curVal : null;
            }
            if ($this->VillageCode->AdvancedSearch->ViewValue !== null) { // Load from cache
                $this->VillageCode->EditValue = array_values($this->VillageCode->lookupOptions());
                if ($this->VillageCode->AdvancedSearch->ViewValue == "") {
                    $this->VillageCode->AdvancedSearch->ViewValue = $Language->phrase("PleaseSelect");
                }
            } else { // Lookup from database
                if ($curVal == "") {
                    $filterWrk = "0=1";
                } else {
                    $filterWrk = SearchFilter($this->VillageCode->Lookup->getTable()->Fields["VillageCode"]->searchExpression(), "=", $this->VillageCode->AdvancedSearch->SearchValue, $this->VillageCode->Lookup->getTable()->Fields["VillageCode"]->searchDataType(), "");
                }
                $sqlWrk = $this->VillageCode->Lookup->getSql(true, $filterWrk, '', $this, false, true);
                $conn = Conn();
                $config = $conn->getConfiguration();
                $config->setResultCache($this->Cache);
                $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                $ari = count($rswrk);
                if ($ari > 0) { // Lookup values found
                    $arwrk = $this->VillageCode->Lookup->renderViewRow($rswrk[0]);
                    $this->VillageCode->AdvancedSearch->ViewValue = $this->VillageCode->displayValue($arwrk);
                } else {
                    $this->VillageCode->AdvancedSearch->ViewValue = $Language->phrase("PleaseSelect");
                }
                $arwrk = $rswrk;
                $this->VillageCode->EditValue = $arwrk;
            }
            $this->VillageCode->PlaceHolder = RemoveHtml($this->VillageCode->caption());

            // Village
            $this->Village->setupEditAttributes();
            if (!$this->Village->Raw) {
                $this->Village->AdvancedSearch->SearchValue = HtmlDecode($this->Village->AdvancedSearch->SearchValue);
            }
            $this->Village->EditValue = HtmlEncode($this->Village->AdvancedSearch->SearchValue);
            $this->Village->PlaceHolder = RemoveHtml($this->Village->caption());

            // NRC
            $this->NRC->setupEditAttributes();
            if (!$this->NRC->Raw) {
                $this->NRC->AdvancedSearch->SearchValue = HtmlDecode($this->NRC->AdvancedSearch->SearchValue);
            }
            $this->NRC->EditValue = HtmlEncode($this->NRC->AdvancedSearch->SearchValue);
            $this->NRC->PlaceHolder = RemoveHtml($this->NRC->caption());

            // FarmerName
            $this->FarmerName->setupEditAttributes();
            if (!$this->FarmerName->Raw) {
                $this->FarmerName->AdvancedSearch->SearchValue = HtmlDecode($this->FarmerName->AdvancedSearch->SearchValue);
            }
            $this->FarmerName->EditValue = HtmlEncode($this->FarmerName->AdvancedSearch->SearchValue);
            $this->FarmerName->PlaceHolder = RemoveHtml($this->FarmerName->caption());

            // DOB
            $this->DOB->setupEditAttributes();
            if (!$this->DOB->Raw) {
                $this->DOB->AdvancedSearch->SearchValue = HtmlDecode($this->DOB->AdvancedSearch->SearchValue);
            }
            $this->DOB->EditValue = HtmlEncode($this->DOB->AdvancedSearch->SearchValue);
            $this->DOB->PlaceHolder = RemoveHtml($this->DOB->caption());

            // Age
            $this->Age->setupEditAttributes();
            $this->Age->EditValue = $this->Age->AdvancedSearch->SearchValue;
            $this->Age->PlaceHolder = RemoveHtml($this->Age->caption());

            // Group
            $this->Group->setupEditAttributes();
            $this->Group->EditValue = $this->Group->options(false);
            $this->Group->PlaceHolder = RemoveHtml($this->Group->caption());

            // Gender
            $this->Gender->setupEditAttributes();
            $curVal = trim(strval($this->Gender->AdvancedSearch->SearchValue));
            if ($curVal != "") {
                $this->Gender->AdvancedSearch->ViewValue = $this->Gender->lookupCacheOption($curVal);
            } else {
                $this->Gender->AdvancedSearch->ViewValue = $this->Gender->Lookup !== null && is_array($this->Gender->lookupOptions()) && count($this->Gender->lookupOptions()) > 0 ? $curVal : null;
            }
            if ($this->Gender->AdvancedSearch->ViewValue !== null) { // Load from cache
                $this->Gender->EditValue = array_values($this->Gender->lookupOptions());
            } else { // Lookup from database
                if ($curVal == "") {
                    $filterWrk = "0=1";
                } else {
                    $filterWrk = SearchFilter($this->Gender->Lookup->getTable()->Fields["Gender"]->searchExpression(), "=", $this->Gender->AdvancedSearch->SearchValue, $this->Gender->Lookup->getTable()->Fields["Gender"]->searchDataType(), "");
                }
                $sqlWrk = $this->Gender->Lookup->getSql(true, $filterWrk, '', $this, false, true);
                $conn = Conn();
                $config = $conn->getConfiguration();
                $config->setResultCache($this->Cache);
                $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                $ari = count($rswrk);
                $arwrk = $rswrk;
                $this->Gender->EditValue = $arwrk;
            }
            $this->Gender->PlaceHolder = RemoveHtml($this->Gender->caption());

            // Seedlings
            $this->Seedlings->setupEditAttributes();
            $this->Seedlings->EditValue = $this->Seedlings->AdvancedSearch->SearchValue;
            $this->Seedlings->PlaceHolder = RemoveHtml($this->Seedlings->caption());

            // Seeds
            $this->Seeds->setupEditAttributes();
            $this->Seeds->EditValue = $this->Seeds->AdvancedSearch->SearchValue;
            $this->Seeds->PlaceHolder = RemoveHtml($this->Seeds->caption());

            // Type
            $this->Type->setupEditAttributes();
            $curVal = trim(strval($this->Type->AdvancedSearch->SearchValue));
            if ($curVal != "") {
                $this->Type->AdvancedSearch->ViewValue = $this->Type->lookupCacheOption($curVal);
            } else {
                $this->Type->AdvancedSearch->ViewValue = $this->Type->Lookup !== null && is_array($this->Type->lookupOptions()) && count($this->Type->lookupOptions()) > 0 ? $curVal : null;
            }
            if ($this->Type->AdvancedSearch->ViewValue !== null) { // Load from cache
                $this->Type->EditValue = array_values($this->Type->lookupOptions());
            } else { // Lookup from database
                if ($curVal == "") {
                    $filterWrk = "0=1";
                } else {
                    $filterWrk = SearchFilter($this->Type->Lookup->getTable()->Fields["SeedVariety"]->searchExpression(), "=", $this->Type->AdvancedSearch->SearchValue, $this->Type->Lookup->getTable()->Fields["SeedVariety"]->searchDataType(), "");
                }
                $sqlWrk = $this->Type->Lookup->getSql(true, $filterWrk, '', $this, false, true);
                $conn = Conn();
                $config = $conn->getConfiguration();
                $config->setResultCache($this->Cache);
                $rswrk = $conn->executeCacheQuery($sqlWrk, [], [], $this->CacheProfile)->fetchAll();
                $ari = count($rswrk);
                $arwrk = $rswrk;
                $this->Type->EditValue = $arwrk;
            }
            $this->Type->PlaceHolder = RemoveHtml($this->Type->caption());

            // DateAllocated
            $this->DateAllocated->setupEditAttributes();
            if (!$this->DateAllocated->Raw) {
                $this->DateAllocated->AdvancedSearch->SearchValue = HtmlDecode($this->DateAllocated->AdvancedSearch->SearchValue);
            }
            $this->DateAllocated->EditValue = HtmlEncode($this->DateAllocated->AdvancedSearch->SearchValue);
            $this->DateAllocated->PlaceHolder = RemoveHtml($this->DateAllocated->caption());

            // ID
            $this->ID->setupEditAttributes();
            $this->ID->EditValue = $this->ID->AdvancedSearch->SearchValue;
            $this->ID->PlaceHolder = RemoveHtml($this->ID->caption());
        }
        if ($this->RowType == RowType::ADD || $this->RowType == RowType::EDIT || $this->RowType == RowType::SEARCH) { // Add/Edit/Search row
            $this->setupFieldTitles();
        }

        // Call Row Rendered event
        if ($this->RowType != RowType::AGGREGATEINIT) {
            $this->rowRendered();
        }
    }

    // Validate search
    protected function validateSearch()
    {
        // Check if validation required
        if (!Config("SERVER_VALIDATE")) {
            return true;
        }
        if (!CheckInteger($this->SupplierCode->AdvancedSearch->SearchValue)) {
            $this->SupplierCode->addErrorMessage($this->SupplierCode->getErrorMessage(false));
        }
        if (!CheckNumber($this->Age->AdvancedSearch->SearchValue)) {
            $this->Age->addErrorMessage($this->Age->getErrorMessage(false));
        }
        if (!CheckNumber($this->Seedlings->AdvancedSearch->SearchValue)) {
            $this->Seedlings->addErrorMessage($this->Seedlings->getErrorMessage(false));
        }
        if (!CheckInteger($this->ID->AdvancedSearch->SearchValue)) {
            $this->ID->addErrorMessage($this->ID->getErrorMessage(false));
        }

        // Return validate result
        $validateSearch = !$this->hasInvalidFields();

        // Call Form_CustomValidate event
        $formCustomError = "";
        $validateSearch = $validateSearch && $this->formCustomValidate($formCustomError);
        if ($formCustomError != "") {
            $this->setFailureMessage($formCustomError);
        }
        return $validateSearch;
    }

    // Load advanced search
    public function loadAdvancedSearch()
    {
        $this->DistrictCode->AdvancedSearch->load();
        $this->DistrictName->AdvancedSearch->load();
        $this->GRNCode->AdvancedSearch->load();
        $this->GRN->AdvancedSearch->load();
        $this->SupplierCode->AdvancedSearch->load();
        $this->Supplier->AdvancedSearch->load();
        $this->CampCode->AdvancedSearch->load();
        $this->Camp->AdvancedSearch->load();
        $this->VillageCode->AdvancedSearch->load();
        $this->Village->AdvancedSearch->load();
        $this->NRC->AdvancedSearch->load();
        $this->FarmerName->AdvancedSearch->load();
        $this->DOB->AdvancedSearch->load();
        $this->Age->AdvancedSearch->load();
        $this->Group->AdvancedSearch->load();
        $this->Gender->AdvancedSearch->load();
        $this->Seedlings->AdvancedSearch->load();
        $this->Seeds->AdvancedSearch->load();
        $this->Type->AdvancedSearch->load();
        $this->DateAllocated->AdvancedSearch->load();
        $this->ID->AdvancedSearch->load();
    }

    // Set up Breadcrumb
    protected function setupBreadcrumb()
    {
        global $Breadcrumb, $Language;
        $Breadcrumb = new Breadcrumb("index");
        $url = CurrentUrl();
        $Breadcrumb->add("list", $this->TableVar, $this->addMasterUrl("CidpDataList"), "", $this->TableVar, true);
        $pageId = "search";
        $Breadcrumb->add("search", $pageId, $url);
    }

    // Setup lookup options
    public function setupLookupOptions($fld)
    {
        if ($fld->Lookup && $fld->Lookup->Options === null) {
            // Get default connection and filter
            $conn = $this->getConnection();
            $lookupFilter = "";

            // No need to check any more
            $fld->Lookup->Options = [];

            // Set up lookup SQL and connection
            switch ($fld->FieldVar) {
                case "x_DistrictCode":
                    break;
                case "x_GRNCode":
                    break;
                case "x_SupplierCode":
                    break;
                case "x_CampCode":
                    break;
                case "x_VillageCode":
                    break;
                case "x_Group":
                    break;
                case "x_Gender":
                    break;
                case "x_Type":
                    break;
                default:
                    $lookupFilter = "";
                    break;
            }

            // Always call to Lookup->getSql so that user can setup Lookup->Options in Lookup_Selecting server event
            $sql = $fld->Lookup->getSql(false, "", $lookupFilter, $this);

            // Set up lookup cache
            if (!$fld->hasLookupOptions() && $fld->UseLookupCache && $sql != "" && count($fld->Lookup->Options) == 0 && count($fld->Lookup->FilterFields) == 0) {
                $totalCnt = $this->getRecordCount($sql, $conn);
                if ($totalCnt > $fld->LookupCacheCount) { // Total count > cache count, do not cache
                    return;
                }
                $rows = $conn->executeQuery($sql)->fetchAll();
                $ar = [];
                foreach ($rows as $row) {
                    $row = $fld->Lookup->renderViewRow($row, Container($fld->Lookup->LinkTable));
                    $key = $row["lf"];
                    if (IsFloatType($fld->Type)) { // Handle float field
                        $key = (float)$key;
                    }
                    $ar[strval($key)] = $row;
                }
                $fld->Lookup->Options = $ar;
            }
        }
    }

    // Page Load event
    public function pageLoad()
    {
        //Log("Page Load");
    }

    // Page Unload event
    public function pageUnload()
    {
        //Log("Page Unload");
    }

    // Page Redirecting event
    public function pageRedirecting(&$url)
    {
        // Example:
        //$url = "your URL";
    }

    // Message Showing event
    // $type = ''|'success'|'failure'|'warning'
    public function messageShowing(&$msg, $type)
    {
        if ($type == "success") {
            //$msg = "your success message";
        } elseif ($type == "failure") {
            //$msg = "your failure message";
        } elseif ($type == "warning") {
            //$msg = "your warning message";
        } else {
            //$msg = "your message";
        }
    }

    // Page Render event
    public function pageRender()
    {
        //Log("Page Render");
    }

    // Page Data Rendering event
    public function pageDataRendering(&$header)
    {
        // Example:
        //$header = "your header";
    }

    // Page Data Rendered event
    public function pageDataRendered(&$footer)
    {
        // Example:
        //$footer = "your footer";
    }

    // Page Breaking event
    public function pageBreaking(&$break, &$content)
    {
        // Example:
        //$break = false; // Skip page break, or
        //$content = "<div style=\"break-after:page;\"></div>"; // Modify page break content
    }

    // Form Custom Validate event
    public function formCustomValidate(&$customError)
    {
        // Return error message in $customError
        return true;
    }
}
